/*
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC.
 */

using Plugin.Firebase.Firestore;

using DXLib.Data;
using DXLib.Utils;

namespace iStatVball3;

/*
 * Encapsulates all custom RallyFlow level settings for either the primary team or opponent. 
 */
public class SettingsCustom : IFirestoreObject
{
	/* Constants */

	/* Properties */
	[FirestoreProperty("QuickKeys")] public IList<string> QuickKeys { get; set; }
	[FirestoreProperty("RatingKeys")] public IList<string> RatingKeys { get; set; }
	[FirestoreProperty("ModifierKeys")] public IList<string> ModifierKeys { get; set; }
	[FirestoreProperty("FaultKeys")] public IList<string> FaultKeys { get; set; }
	
	/* Methods */
	public SettingsCustom()
	{
		QuickKeys = new List<string>();
		RatingKeys = new List<string>();
		ModifierKeys = new List<string>();
		FaultKeys = new List<string>();
	}
	
	// Adds individual option to custom option list with specified key
	public void AddOption( string key, string option )
	{
		GetList( key )?.Add( option );
	}

	// Removes individual option from custom option list with specified key
	public void RemoveOption( string key, string option )
	{
		GetList( key )?.Remove( option );
	}

	// Clears all options from all lists
	public void ClearAll()
	{
		QuickKeys.Clear();
		RatingKeys.Clear();
		ModifierKeys.Clear();
		FaultKeys.Clear();
	}
	
	// Turns ON all custom options of specified type
	public void TurnAllOn( string key, string baseKey )
	{
		IList<string> list = GetList( key );

		// Lookup option list
		List<DXItem> options = DXString.GetLookupList( baseKey );
			
		// Add to custom list
		foreach ( DXItem option in options )
		{
			list.Add( option.Key );
		}
	}

	// Turns OFF all custom options of specified type
	public void TurnAllOff( string key )
	{
		Clear( key );
	}
	
	// Clears all options from specified list
	private void Clear( string key )
	{
		GetList( key )?.Clear();
	}
	
	// Determines if specified list contains given value
	public bool Contains( string key, string value )
	{
		IList<string> list = GetList( key );
		
		return list.Contains( value );
	}
	
	// Returns list matching specified type
	public IList<string> GetList( string key )
	{
		return key switch
		{
			Settings.QuickKey => QuickKeys,
			Settings.RatingKey => RatingKeys,
			Settings.ModifierKey => ModifierKeys,
			Settings.FaultKey => FaultKeys,
			
			_ => null
		};
	}
	
	// Sets new list values for list matching specified key
	public void SetList( string key, List<string> values )
	{
		switch ( key )
		{
			case Settings.QuickKey: QuickKeys = new List<string>( values ); break;
			case Settings.RatingKey: RatingKeys = new List<string>( values ); break;
			case Settings.ModifierKey: ModifierKeys = new List<string>( values ); break;
			case Settings.FaultKey: FaultKeys = new List<string>( values ); break;
		}
	}
}

//
