﻿/*
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC.
 */

using Plugin.Firebase.Firestore;

namespace iStatVball3;

/*
 * Encapsulates all app configuration settings for an individual user. These settings will follow the user across
 * devices/platforms. Persisted as a map within a parent User document NOT as a standalone document.
 */ 
public class Settings : IFirestoreObject
{
	/* Constants */

	// Recording engine
	public const string RallyEngine = "rally";
	public const string LegacyEngine = "legacy";

	// Db write frequency
	public const string WriteAction = "action";
	public const string WriteRally = "rally";

	// Pass Ratings
	public const string Pass03Key = "pass03";
	public const string Pass04Key = "pass04";

	// Side switch options
	public const string SwitchSets = "sets";
	public const string SwitchNever = "never";

	// RallyFlow levels
	public const string LowKey = "low";
	public const string MediumKey = "med";
	public const string HighKey = "high";
	public const string MaximumKey = "max";
	public const string CustomKey = "custom";
	
	// RallyFlow custom option keys
	public const string QuickKey = "quick";
	public const string RatingKey = "rate";
	public const string ModifierKey = "mod";
	public const string FaultKey = "fault";

	// RallyFlow custom option LUTs
	public const string CustomAction = "action";
	public const string CustomRating = "action.rating";

	// Smart Locations
	public const string LiberoLBKey = "left";
	public const string LiberoMBKey = "middle";

	// Name display
	public const string FirstLastKey = "firstlast";
	public const string LastFirstKey = "lastfirst";
	public const string NumberNameKey = "numname";

	// Roster sorting
	public const string SortLastKey = "last";
	public const string SortFirstKey = "first";
	public const string SortNumberKey = "number";

	// Export modes
	public const string EmailLocal = "local";
	public const string EmailCloud = "cloud";

	// Export format
	public const string FormatPdf = "pdf";
	public const string FormatExcel = "excel";

	// Export MaxPreps
	public const string MaxPrepsAuto = "auto";
	public const string MaxPrepsFile = "file";

    /* Properties */

    // Recording
	[FirestoreProperty("RecordEngine")] public string RecordEngine { get; set; }
	[FirestoreProperty("RecordPass")] public string RecordPass { get; set; }
    [FirestoreProperty("RecordSwitch")] public string RecordSwitch { get; set; }
    [FirestoreProperty("RecordWrite")] public string RecordWrite { get; set; }

    // RallyFlow
    [FirestoreProperty("RallyLevel")] public string RallyLevel { get; set; }
    [FirestoreProperty("RallyAuto")] public bool RallyAuto { get; set; }
    [FirestoreProperty("RallyPreview")] public bool RallyPreview { get; set; }

    // Custom Options (RallyFlow)
	[FirestoreProperty("RallyCustom")] public SettingsCustom RallyCustom { get; set; }
    [FirestoreProperty("RallyCustomOpp")] public SettingsCustom RallyCustomOpp { get; set; }

    // Player Grid (RallyFlow)
    [FirestoreProperty("GridDisplay")] public string GridDisplay { get; set; }
    [FirestoreProperty("GridOrder")] public string GridOrder { get; set; }
    [FirestoreProperty("GridTitle")] public bool GridTitle { get; set; }

    // Legacy
    [FirestoreProperty("LegacyFocus")] public bool LegacyFocus { get; set; }
    [FirestoreProperty("LegacySmart")] public bool LegacySmart { get; set; }

    [FirestoreProperty("LegacySet")] public string LegacySet { get; set; }
    [FirestoreProperty("LegacyServe")] public string LegacyServe { get; set; }

    [FirestoreProperty("LegacyPass0")] public bool LegacyPass0 { get; set; }
    [FirestoreProperty("LegacyRecv")] public bool LegacyRecv { get; set; }

    // Smart Locations
    [FirestoreProperty("SmartLibero")] public string SmartLibero { get; set; }
    [FirestoreProperty("SmartAI")] public bool SmartAI { get; set; }

    // Substitutions
    [FirestoreProperty("SubsMax")] public int SubsMax { get; set; }
    [FirestoreProperty("SubsWarn")] public int SubsWarn { get; set; }

    // Timeouts
    [FirestoreProperty("TimeoutMax")] public int TimeoutMax { get; set; }
    [FirestoreProperty("TimeoutSec")] public int TimeoutSec { get; set; }

    // Analyze
    [FirestoreProperty("AnalyzePlayer")] public string AnalyzePlayer { get; set; }
    [FirestoreProperty("AnalyzeDigits")] public int AnalyzeDigits { get; set; }
    [FirestoreProperty("AnalyzeScrimmage")] public bool AnalyzeScrimmage { get; set; }

    // Export
    [FirestoreProperty("ExportEmail")] public string ExportEmail { get; set; }
    [FirestoreProperty("ExportFormat")] public string ExportFormat { get; set; }
    [FirestoreProperty("ExportMaxPreps")] public string ExportMaxPreps { get; set; }

    // General
    [FirestoreProperty("GeneralSort")] public string GeneralSort { get; set; }
    [FirestoreProperty("GeneralAdd")] public string GeneralAdd { get; set; }
    [FirestoreProperty("GeneralFilter")] public bool GeneralFilter { get; set; }
    [FirestoreProperty("GeneralSample")] public bool GeneralSample { get; set; }

	/* Ignored */
	public bool IsRally => RecordEngine == RallyEngine;
	public bool IsWriteAction => RecordWrite == WriteAction;

	public bool IsPass03 => RecordPass == Pass03Key;
	public bool IsPass04 => RecordPass == Pass04Key;

	public bool IsSwitchSides => RecordSwitch != SwitchNever;

	public bool IsLowLevel => RallyLevel == LowKey;
	public bool IsCustomLevel => RallyLevel == CustomKey;

	public bool IsUnlimitedSubs => SubsMax == 0;
	public bool IsLocalEmail => ExportEmail == EmailLocal;

	/* Methods */
	public Settings()
	{
		// MUST set defaults for backward compatibility with new settings
		SetDefaults();
	}

	// (Re)write default values for all settings
	public void SetDefaults()
	{
		// Recording
		RecordEngine = RallyEngine;
		RecordPass = Pass03Key;
		RecordSwitch = SwitchSets;
		RecordWrite = WriteRally;

		// RallyFlow
		RallyLevel = MediumKey;
		RallyAuto = true;
		RallyPreview = true;

		// Custom Options (RallyFlow)
		RallyCustom = new SettingsCustom();
		RallyCustomOpp = new SettingsCustom();

		SetLevel( RallyLevel );

		// Player Grid (RallyFlow)
		GridDisplay = "first";
		GridOrder = "smart";
		GridTitle = false;

		// Legacy
		LegacyFocus = true;
		LegacySmart = false;

		LegacySet = "off";
		LegacyServe = "aae";

		LegacyPass0 = true;
		LegacyRecv = false;

		// Smart Locations
		SmartLibero = LiberoLBKey;
		SmartAI = true;

		// Substitutions
		SubsMax = 18;
		SubsWarn = 2;

		// Timeouts
		TimeoutMax = 2;
		TimeoutSec = 30;

		// Analyze
		AnalyzePlayer = FirstLastKey;
		AnalyzeDigits = 1;
		AnalyzeScrimmage = false;

		// Export
		ExportEmail = EmailCloud;
		ExportFormat = FormatPdf;
		ExportMaxPreps = MaxPrepsAuto;

		// General
		GeneralSort = SortLastKey;
		GeneralAdd = "end";
		GeneralFilter = false;
		GeneralSample = true;
	}

	/* RallyFlow Custom */
	
	// Sets custom option settings based on specified detail level
	public void SetLevel( string level )
	{
		// Do NOT call for Custom level
		RallyCustom.ClearAll();
		RallyCustomOpp.ClearAll();
	
		// Add options for level
		switch ( level )
		{
			case LowKey: SetCustomLow(); break;
			case MediumKey: SetCustomMedium(); break;
			case HighKey: SetCustomHigh(); break;
			case MaximumKey: SetCustomMax(); break;
		}
	}
	
	// LOW detail level
	private void SetCustomLow()
	{
		RallyAuto = true;
	
		// All QuickSelect ON for both teams
		RallyCustom.TurnAllOn( QuickKey, CustomAction );
		RallyCustomOpp.TurnAllOn( QuickKey, CustomAction );
	
		// All Ratings OFF for both teams
		RallyCustom.TurnAllOff( RatingKey );
		RallyCustomOpp.TurnAllOff( RatingKey );
	
		// All Modifiers OFF for both teams
		RallyCustom.TurnAllOff( ModifierKey );
		RallyCustomOpp.TurnAllOff( ModifierKey );
	
		// All Faults OFF for both teams
		RallyCustom.TurnAllOff( FaultKey );
		RallyCustomOpp.TurnAllOff( FaultKey );
	}
	
	// MEDIUM detail level
	private void SetCustomMedium()
	{
		SetCustomLow();
	
		// Only some QuickSelect ON for primary team
		RallyCustom.TurnAllOff( QuickKey );
	
		RallyCustom.AddOption( QuickKey, Stats.ServeKey );
		RallyCustom.AddOption( QuickKey, Stats.SecondKey );
		RallyCustom.AddOption( QuickKey, Stats.PutbackKey );
		RallyCustom.AddOption( QuickKey, Stats.BlockKey );
	}
	
	// HIGH detail level
	private void SetCustomHigh()
	{
		SetCustomMedium();
	
		RallyAuto = false;
	
		// Some ratings ON for primary teams
		RallyCustom.AddOption( RatingKey, Stats.ServeKey );
		RallyCustom.AddOption( RatingKey, Stats.ReceiveKey );
		RallyCustom.AddOption( RatingKey, Stats.FirstKey );
		RallyCustom.AddOption( RatingKey, Stats.DefenseKey );
		RallyCustom.AddOption( RatingKey, Stats.FreeballKey );
	
		// 2nd Ball QuickSelect OFF for pass/serve ratings
		RallyCustom.RemoveOption( QuickKey, Stats.SecondKey );
	
		// All Faults ON for primary team
		RallyCustom.TurnAllOn( FaultKey, CustomAction );
	}
	
	// MAXIMUM detail level
	private void SetCustomMax()
	{
		RallyAuto = false;
	
		// All QuickSelect OFF for both teams
		RallyCustom.TurnAllOff( QuickKey );
		RallyCustomOpp.TurnAllOff( QuickKey );
	
		// All Ratings ON for both teams
		RallyCustom.TurnAllOn( RatingKey, CustomRating );
		RallyCustomOpp.TurnAllOn( RatingKey, CustomRating );
	
		// All Modifiers ON for both teams
		RallyCustom.TurnAllOn( ModifierKey, CustomAction );
		RallyCustomOpp.TurnAllOn( ModifierKey, CustomAction );
	
		// All Faults ON for both teams
		RallyCustom.TurnAllOn( FaultKey, CustomAction );
		RallyCustomOpp.TurnAllOn( FaultKey, CustomAction );
	}
}

//
