﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;
using DXLib.UI.Layout;
using DXLib.UI.Container;

using DXLib.UI.Control;
using DXLib.UI.Control.Image;
using DXLib.UI.Control.Badge;
using DXLib.UI.Control.Button;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Implements a small, non-modal popup displaying the current root career stats link for a Player. A remove button
 * allows the user to remove the root association.
 */
public class RootRemover
{
	/* Events */
	public Action Removed { get; set; }
	public Action Closed { get; set; }

	/* Properties */
	public bool HasChanges { get; set; }

	/* Fields */
	private readonly DXPopup popup;

	private readonly DXImageArea avatarImg;
	private readonly DXNumberBadge numberBadge;

	private readonly DXLabel nameLbl;
	private readonly DXLabel teamLbl;

	/* Methods */
	public RootRemover()
	{
		// Underlying layout
		DXGridLayout layout = new()
		{
			Padding = 0,
			RowSpacing = 0,
			ColumnSpacing = 10
		};

		const double avatarSize = 60;
		double labelSize = (avatarSize / 2);
		const double numberSize = 35;

		const double pad = 10;

		// 3 rows
		layout.AddFixedRow( labelSize );				// 0: name
		layout.AddFixedRow( labelSize );				// 1: team/season
		layout.AddStarRow();							// 2: button

		// 3 columns
		layout.AddFixedColumn( avatarSize );			// 0: avatar
		layout.AddStarColumn();							// 1: name
		layout.AddFixedColumn( numberSize + pad );		// 2: number

		// Avatar
		avatarImg = new DXImageArea
		{
			Color = DXColors.Action,
			DefaultIcon = "player"
		};

		avatarImg.SetImage( (string)null );
		avatarImg.SetSize( avatarSize, avatarSize, 0.80 );

		layout.Add( avatarImg, 0, 0, 1, 2 );

		// Name
		nameLbl = new DXLabel
		{
			TextColor = DXColors.Dark1,
			Font = DXFonts.RobotoBold,
			FontSize = 17,

			LineBreakMode = LineBreakMode.TailTruncation,
			Vertical = LayoutOptions.Center
		};

		layout.Add( nameLbl, 1, 0 );

		// Team/Season
		teamLbl = new DXLabel
		{
			TextColor = DXColors.Dark1,
			Font = DXFonts.Roboto,
			FontSize = 15,

			LineBreakMode = LineBreakMode.TailTruncation,
			Vertical = LayoutOptions.Start
		};

		layout.Add( teamLbl, 1, 1 );

		// Number 
		numberBadge = new DXNumberBadge
		{
			Color = DXColors.Action,
			TextColor = DXColors.Light4,

			Size = numberSize,
			FontSize = 20,
			AdjustX = 1,

			Horizontal = LayoutOptions.Start,
			Vertical = LayoutOptions.Center
		};

		numberBadge.Init();

		layout.Add( numberBadge, 2, 0, 1, 2 );

		// Remove
		DXButton removeBtn = new()
		{
			Resource = "root.remove",
			Type = DXButton.ButtonType.Negative,

			Horizontal = LayoutOptions.Center,
			Vertical = LayoutOptions.Center,

			ButtonWd = 80,
			Margin = new Thickness( 5, 5, 0, 0 ),

			ButtonTapped = OnRemoveTapped
		};

		layout.Fill( DXColors.Light1, 0, 2, 3, 1 );
		layout.Add( removeBtn, 0, 2, 3, 1 );

		// Non-modal popup
		popup = new DXPopup( DXString.Get( "root.title" ), layout )
		{
			ViewWidth = 285,
			ViewHeight = 110,

			PopupClosed = OnClosed
		};
	}

	// Post construction initialization
	public void Init( Root root )
	{
		// FKs MUST be pre-populated
		Player player = root.Player;

		// Populate UI components
		avatarImg.SetImage( player.ImageUrl );
		nameLbl.Text = player.FullName;
		numberBadge.Text = player.Number;

		Season season = root.Season;

		// 'Varsity 2019'
		teamLbl.Text = $"{season.Team.Name} {season.Name}";
	}

	// Show popup menu
	public void Show( View view )
	{
		HasChanges = false;

		popup.ShowFromView( view );
	}

	// Hide popup menu
	public void Dismiss()
	{
		popup.Dismiss();
	}

	/* Event Callbacks */

	// User tapped remove, callback listener
	private void OnRemoveTapped( object sender )
	{
		HasChanges = true;

		Removed?.Invoke();
	}

	// User cancelled remove
	private void OnClosed()
	{
		Closed?.Invoke();
	}
}

//
