﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using Plugin.Firebase.Firestore;

namespace iStatVball3;

/*
 * Encapsulates all fields necessary to link an entity (Player, Opponent, etc) to an entity of the same type from a
 * previous season. Used for importing previous season data and for analyzing career stats. Persisted as a map in the
 * parent entity, NOT as a root level collection.
 */
public class Root : IFirestoreObject
{
	/* Properties */

	// Required
	public string RootId { get; set; }

	// Optional (Player only)
	public string TeamId { get; set; }
	public string SeasonId { get; set; }
	public string ObjectId { get; set; }

	/* Ignored */

	// References (Player only)
	public Team Team { get; set; }
	public Season Season { get; set; }
	public Player Player { get; set; }

	/* Methods */

	// Determines if required fields populated and link valid (Player only)
	public async Task<bool> IsValid( Organization org )
    {
		bool valid = (RootId != null) && (TeamId != null) && (SeasonId != null) && (ObjectId != null);

		// Make sure player still exists
		if ( valid )
		{
			await Populate( org );
		}

		return Player != null;
    }

	// Populates convenience references from FKs (Player only)
	public async Task Populate( Organization org )
	{
		Team = org.GetTeam( TeamId );
		Season = Team.GetSeason( SeasonId );

		// Roster may not be populated yet
		await Season.PopulateRoster();
		Player = Season.GetPlayer( ObjectId );
	}
}

//
