﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI.Card;
using DXLib.Utils;

namespace iStatVball3;

/*
 * Card page for list of players within a parent season.
 */
public class PlayerPage : CardPage
{
	/* Methods */
	public PlayerPage( Season season ) : base( season )
	{
		Title = DXString.Get( "player.roster" );
	}

	/* Abstracts */

	// Displays all player cards
	public override void LoadCards()
	{
		Clear();
        
        // Required for permissions
        User user = Shell.CurrentUser;

		if ( data is Season season )
		{
			List<Player> players = season.GetPlayers();

			// Add card for each player
			foreach ( Player player in players )
			{
				PlayerCard card = new( player )
				{
					DefaultIcon = "player",
					ImageUrl = player.ImageUrl,
					ImageColor = season.Color,

					Title = $"{player.FullName}",			// WORKAROUND: Android crash
					SubTitle = player.PositionList(),

					Number = player.Number,
					AltNumber = player.AltNumber,

					LinkStatus = Link.GetMultiStatus( player.Link, player.FanLinks ),
					RootStatus = player.IsRooted,

					HasAnalyze = false,

					IsEditEnabled = Player.CanEdit( user ),
					IsAnalyzeEnabled = player.CanAnalyze( user ),

					IsTappable = false,
					AnalyzeTapped = OnAnalyzeTapped,
					EditTapped = OnEditTapped
				};

				card.Init();

				// Deactivated players are faded
				card.SetDeactivated( player.IsDeactivated );

				AddCard( card );
			}

			// Add card for creating new objects
			if ( Player.CanCreate( season, user ) )
			{
				AddNewCard( "player", CardSmall.CardHt );
			}
		}
       
    }

	/* Event Callbacks */

	// Called back when 'Add New' tapped
	protected override void OnNewCardTapped( DXCard card )
	{
		Season season = data as Season;
		PlayerForm form = new( null, season );

		// Open in create mode
		Shell.Instance.ShowForm( form );
	}

	// Called back when card analyze button tapped
	private void OnAnalyzeTapped( DXCard card )
	{}

	// Called back when card edit button tapped
	private void OnEditTapped( DXCard card )
	{
		Player player = card.Data as Player;
		Season season = data as Season;

		PlayerForm form = new( player, season );

		// Open in edit mode
		Shell.Instance.ShowForm( form );
	}
}

//
