﻿/*
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC.
 */

using DXLib.UI;
using DXLib.UI.Card;
using DXLib.UI.Alert;

using DXLib.Data;
using DXLib.Utils;

namespace iStatVball3;

/*
 * Card page for list of organizations.
 */
public class OrganizationPage : CardPage
{
	/* Methods */
	public OrganizationPage( User user ) : base( user )
	{
		Title = DXString.Get( "organization.plural" );
	}

	/* Abstracts */

	// Displays all organization cards
	public override void LoadCards()
	{
		Clear();

		// Required for permissions
		User user = data as User;

		if ( user != null )
		{
			bool admin = user.IsAdmin;
	
			List<Organization> organizations = Organization.CanView( user );
	
			// Add card for each org
			foreach ( Organization org in organizations )
			{
				// Might be hiding sample org
				if ( (org == null) || (org.IsSample && !user.Settings.GeneralSample) )
				{
					continue;
				}

				// Card might be locked
				bool readOnly = org.IsSample && !admin;

				// 'SAMPLE' or org type
				string superTitle = org.IsSample ? DXString.Get( "organization.sample" ) : org.TypeName.ToUpper();

				OrganizationCard card = new( org, readOnly )
				{
					Data = org,

					DefaultIcon = "organization",
					ImageUrl = org.ImageUrl,
					ImageColor = org.Color,

					SuperTitle = superTitle,
					Title = org.Name,
					SubTitle = org.Mascot,

					Count = org.Teams.Count,
					CountTitle = "team",

					HasAnalyze = true,

					IsAnalyzeEnabled = org.CanAnalyze( user ),
					IsEditEnabled = org.CanEdit( user ),

					CardTapped = OnCardTapped,
					AnalyzeTapped = OnAnalyzeTapped,
					EditTapped = OnEditTapped
				};
				
				card.Init();
				
				// Visually indicate debug orgs
				if ( org.IsDebug && admin )
				{
					card.Color = DXColors.Accent3;
				}

				AddCard( card );
			}
		}

		// Add card for creating new objects (only if allowed)
		if ( Organization.CanCreate( user ) )
		{
			AddNewCard( "organization", CardLarge.CardHt );
		}
	}

	/* Event Callbacks */

	// Called back when 'Add New' tapped
	protected override void OnNewCardTapped( DXCard card )
	{
		// Clear any previous scope (MUST be first)
		Organization.SetScope( null );

		User user = data as User;
		OrganizationForm form = new( null, user );

		// Open in create mode
		Shell.Instance.ShowForm( form );
	}

	// Called back when normal card tapped
	private static void OnCardTapped( DXCard card )
	{
		Organization organization = card.Data as Organization;
		TeamPage page = new( organization );

		// Navigating down org tree
		Organization.SetScope( organization );

		// Navigate to teams page for selected org
		Shell.Instance.PageStack.Push( page, organization?.Name );
	}

	// Called back when card analyze button tapped
	private static void OnAnalyzeTapped( DXCard card )
	{
		// Network connection required for career stats
		if ( !DXData.HasConnection() )
		{
			DXAlert.ShowNetworkError( "net.err.career", card.Reset );
		}
		// User already warned about performance
		else if ( DXPreferences.Get( "org.warned", false ) )
		{
			OnAnalyzeConfirmed( card );
		}
		// Warn user
		else
		{
			DXAlert.ShowOkCancel( "analyze.perf", "analyze.perf.msg", () => OnAnalyzeConfirmed( card ), card.Reset );

			// Only warn once
			DXPreferences.Set( "org.warned", true );
		}
	}

	// Called back when card analyze button tapped
	private static async void OnAnalyzeConfirmed( DXCard card )
	{
		if ( card.Data is Organization org )
		{
			Shell shell = Shell.Instance;

			// Show dashboard metrics
			shell.DashboardForm.Init( new DashboardConfig
			{
				Scope = DataConfig.OrganizationScope,
				ScopeName = org.Name,
				ScopeObject = org,

				Organization = org,

				Team1 = null,
				Team2 = null
			});

			await shell.ShowDashboardForm( org.Name );
		}
	}

	// Called back when card edit button tapped
	private void OnEditTapped( DXCard card )
	{
		Organization org = card.Data as Organization;
		User user = data as User;

		// Editing considered navigating down org tree
		Organization.SetScope( org );

		OrganizationForm form = new( org, user );

		// Open in edit mode
		Shell.Instance.ShowForm( form );
	}
}

//
