﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI.Card;
using DXLib.Utils;

namespace iStatVball3;

/*
 * Card page for list of Opponents within a parent Season.
 */
public class OpponentPage : CardPage
{
	/* Methods */
	public OpponentPage( Season season ) : base( season )
	{
		Title = DXString.Get( "opponent.plural" );
	}

	/* Abstracts */

	// Displays all opponent cards
	public override void LoadCards()
	{
		Clear();

		// Required for permissions
		User user = Shell.CurrentUser;

		if ( data is Season season )
		{
			var opponents = season.Opponents.OrderBy( o => o.Organization );

			// Add card for each opponent
			foreach ( Opponent opponent in opponents )
			{
				OpponentCard card = new( opponent, false )
				{
					Data = opponent,

					DefaultIcon = "opponent",
					ImageUrl = opponent.ImageUrl,
					ImageColor = opponent.Color,

					Title = opponent.Organization,
					SubTitle = (opponent.Team == Opponent.BlankOpponent) ? null : opponent.Team,

					HasAnalyze = true,

					IsEditEnabled = Opponent.CanEdit( user ),
					IsAnalyzeEnabled = opponent.CanAnalyze( user ),

					IsTappable = false,
					AnalyzeTapped = OnAnalyzeTapped,
					EditTapped = OnEditTapped
				};

				card.Init();
				
				// 'Won-Lost-Tied'
				card.UpdateRecord( opponent.Won, opponent.Lost, opponent.Tied );

				AddCard( card );
			}

			// Add card for creating new objects
			if ( Opponent.CanCreate( season, user ) )
			{
				AddNewCard( "opponent", CardSmall.CardHt );
			}
		}
	}

	/* Event Callbacks */

	// Called back when 'Add New' tapped
	protected override void OnNewCardTapped( DXCard card )
	{
		Season season = data as Season;
		OpponentForm form = new( null, season );

		// Open in create mode
		Shell.Instance.ShowForm( form );
	}

	// Called back when card analyze button tapped
	private async void OnAnalyzeTapped( DXCard card )
	{
		if ( (card.Data is Opponent opponent) && (data is Season season) )
		{
			Shell shell = Shell.Instance;

			// Show dashboard metrics
			shell.DashboardForm.Init( new DashboardConfig
			{
				Scope = DataConfig.OpponentScope,
				ScopeName = opponent.FullName,
				ScopeObject = opponent,

				Organization = season.Team.Organization,

				Team1 = season.Team,
				Team2 = null
			});

			await shell.ShowDashboardForm( opponent.FullName );
		}
	}

	// Called back when card edit button tapped
	private void OnEditTapped( DXCard card )
	{
		Opponent opponent = card.Data as Opponent;
		Season season = data as Season;

		OpponentForm form = new( opponent, season );

		// Open in edit mode
		Shell.Instance.ShowForm( form );
	}
}

//
