﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI.Form;
using DXLib.UI.Form.Control;

namespace iStatVball3;

/*
 * CRUD data entry form for an Opponent.
 */
public sealed class OpponentForm : DXForm
{
	/* Fields */
	private readonly Season season;

	// MaxPreps
	private MP2Interface maxPreps;

	/* Methods */
	public OpponentForm( Opponent opponent, Season season ) : base( opponent, "opponent" )
	{
		this.season = season;

		header.Title = HasData ? opponent.FullName : CreateHeader();
		imagePanel.Color = opponent?.Color;

		// May have MaxPreps link
		InitMaxPreps();

		/* Required */

		// Organization
		AddControl( new DXTextField
		{
			Key = "org",
			Title = "opponent.org",
			Text = opponent?.Organization,
			MinLength = 1,
			MaxLength = 64,
			Type = DXTextField.TextType.CapitalizeWords,
			Hint = DXFormControl.HintType.RequiredMin,
			Help = "opponent.org"
		}, true, true );

		// Team
		AddControl( new DXTextField
		{
			Key = "team",
			Title = "opponent.team",
			Text = opponent?.Team,
			MinLength = 1,
			MaxLength = 64,
			Type = DXTextField.TextType.CapitalizeWords,
			Hint = DXFormControl.HintType.RequiredMin,
			Help = "opponent.team"
		}, true, true );

		// Abbreviation
		AddControl( new DXTextField
		{
			Key = "abbrev",
			Title = "form.abbrev",
			Text = opponent?.Abbreviation,
			MinLength = 1,
			MaxLength = 5,
			Type = DXTextField.TextType.AllCaps,
			Hint = DXFormControl.HintType.RequiredRange,
			Help = "opponent.abbrev"
		}, true, true );

		/* Optional */

		// Mascot
		AddControl( new DXTextField
		{
			Key = "mascot",
			Title = "opponent.mascot",
			Text = opponent?.Mascot,
			MaxLength = 32,
			Type = DXTextField.TextType.CapitalizeWords,
			Help = "opponent.mascot"
		}, false, false );

		// Location
		AddControl( new DXTextField
		{
			Key = "location",
			Title = "form.location",
			Text = opponent?.Location,
			MinLength = 1,
			MaxLength = 64,
			Type = DXTextField.TextType.CapitalizeWords,
			Help = "opponent.location"
		}, false, false );

		// Color
		AddControl( new DXColorPickerField
		{
			Key = "color",
			Title = "opponent.color",
			PickedValue = opponent?.ColorValue,
			DefaultColor = Organization.DefaultColor,
			Help = "opponent.color"
		}, false, false );

		// Notes
		AddControl( new DXEditorField
		{
			Key = "notes",
			Title = "form.notes",
			Text = opponent?.Notes,
			MaxLength = 1024,
			Help = "opponent.notes"
		}, false, false );

		// Image
		SetImage( opponent?.ImageUrl );

		// Control initialization
		Init();
	}

	// Initializes MaxPreps link
	private void InitMaxPreps()
	{
		Opponent opponent = data as Opponent;

		// Check permissions
		if ( Opponent.CanCreate( season, Shell.CurrentUser ) && season.IsHighSchool )
		{
			// Create selector/remover
			maxPreps = new MP2Interface()
			{
				Form = this,
				School = opponent?.MaxPreps,
				IsHighSchool = season.IsHighSchool
			};

			// Show link button
			maxPreps.UpdateHeader();
		}
	}

    /* Abstracts */

	// Overrides to include MaxPreps
    protected override bool HasChanges()
    {
        return base.HasChanges() || (maxPreps is { HasChanges: true });
    }

    /* Event Callbacks */

    // User confirmed delete
    protected override async void OnDeleteConfirmed()
	{
		base.OnDeleteConfirmed();

		// Cascading delete
		if ( data is Opponent opponent )
		{
			await opponent.Delete( true );
		}

		// Refresh UI
		Shell.Instance.HideForm();
	}

	// Used confirmed cancel
	protected override void OnCancelConfirmed()
	{
		base.OnCancelConfirmed();

		Shell.Instance.HideForm();
	}

	// Used tapped save
	protected override async void OnSaveTapped()
	{
		base.OnSaveTapped();

		// Retrieve fields
		string organization = GetString( "org", true );
		string team = GetString( "team", true );
		string abbrev = GetString( "abbrev", true );

		string mascot = GetString( "mascot", false );
		string location = GetString( "location", false );
		long color = GetColor( "color", false );
		string notes = GetString( "notes", false );

		MP2School school = maxPreps?.School;

		Opponent opponent;

		// Update existing object
		if ( HasData )
		{
			opponent = data as Opponent;

			if ( opponent != null )
			{
				opponent.Organization = organization;
				opponent.Team = team;
				opponent.Abbreviation = abbrev;

				opponent.Mascot = mascot;
				opponent.Location = location;
				opponent.ColorValue = color;
				opponent.Notes = notes;

				opponent.MaxPreps = school;

				// Image
				await SaveImage( opponent );

				// Persist
				await opponent.Update();
			}
		}
		// Create new object
		else
		{
			opponent = new Opponent
			{
				Organization = organization,
				Team = team,
				Abbreviation = abbrev,

				Mascot = mascot,
				Location = location,
				ColorValue = color,
				Notes = notes,

				MaxPreps = school,

				// Set parent
				SeasonId = season.UniqueId,
				Season = season
			};

			// Image
			await SaveImage( opponent );

			// Add to parent
			season.Opponents.Add( opponent );

			// Persist
			await opponent.Create();
		}

		// Refresh UI
		Shell.Instance.HideForm( opponent );
	}

    // User tapped MaxPreps link button
    protected override async void OnCustomHeaderTapped()
    {
        await maxPreps.Show();
    }
}

//
