﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.Utils;

namespace iStatVball3;

/*
 * UI implementation for viewing, creating, or editing a starting lineup. A grid of lineup cells allows player selection
 * for zones 1-6 plus two optional libero zones.
 */
public class LineupStarter : LineupPanel
{
	/* Constants */
	private const string StarterKey = "starter";

	/* Fields */
	private LineupStarterCell selectedCell;

	// Zones 1-8
	private readonly List<LineupStarterCell> cells;

	/* Methods */

	// NO UI in constructor
	public LineupStarter()
	{
		Type = StarterKey;

		// Allocate container
		cells = new List<LineupStarterCell>( Lineup.MaxEntries );

        // Create underlying cells without UI (zones 1-8)
        for ( int zone = 1; zone <= Lineup.LiberoZone2; zone++ )
        {
            LineupStarterCell cell = new( zone )
			{
				CellTapped = OnCellTapped,
				Changed = OnChanged
			};

            cells.Add( cell );
        }
    }

    // Performs all layout initialization for control
    public override void Init( IList<Player> roster, Color color, bool editable )
	{
        base.Init( roster, color, editable );

		// 1 additional row
		AddStarRow();

		int row = IsEditEnabled ? 3 : 2;

		// Liberos (7-8)
		AddCell( 7, 0, row );
		AddCell( 8, 1, row );

		// Populate
		if ( Roster != null )
		{
			LineupMenu.SetPlayers( Roster );
		}

		IsInitialized = true;
	}

	// Creates individual cell (zones 1-8) in control
	protected override void AddCell( int zone, int col, int row )
	{
		LineupStarterCell cell = cells[ zone - 1 ];

		// Initialize UI
		cell.Init( Color, IsEditEnabled );

		// Put in correct grid position
		Add( cell, col, row );
	}

	/* Entries */

	// Returns list of all current zone entries (player/position/number)
	public List<LineupEntry> GetEntries()
	{
		List<LineupEntry> list = new( Lineup.MaxEntries );

		// Build entry from each cell
		for ( int zone = 0; zone < Lineup.MaxEntries; zone++ )
		{
			LineupStarterCell cell = cells[ zone ];
			Player player = cell.Player;

			LineupEntry entry = new()
			{
				Zone = (zone + 1),

				Position = cell.Position,
				IsAltNumber = cell.IsAltNumber,

				PlayerId = player?.RootId,
				Player = player
			};

			list.Add( entry );
		}

		return list;
	}

	// Populates all entries for this starting lineup
	public void SetEntries( IList<LineupEntry> entries )
	{
		if ( entries != null )
		{
			for ( int zone = 0; zone < Lineup.MaxEntries; zone++ )
			{
				cells[ zone ].SetEntry( entries[ zone ] );
			}
		}
	}

	// Returns list of all players currently in lineup
	public List<Player> GetPlayers()
	{
		List<Player> list = new( Lineup.MaxEntries );

		// Ignore empty cells
		foreach ( LineupStarterCell cell in cells )
		{
			if ( !cell.IsEmpty )
			{
				list.Add( cell.Player );
			}
		}

		return list;
	}

	// Returns libero(s) currently in lineup
	public List<Player> GetLiberos()
	{
		List<Player> list = new( Lineup.MaxLiberos );

		// Libero 1
		LineupStarterCell libero1 = cells[ Lineup.LiberoZone1 - 1 ];

		if ( !libero1.IsEmpty )
		{
			list.Add( libero1.Player );
		}

		// Libero 2
		LineupStarterCell libero2 = cells[ Lineup.LiberoZone2 - 1 ];

		if ( !libero2.IsEmpty )
		{
			list.Add( libero2.Player );
		}

		return list;
	}

    /* Update */

    // Forces redraw of all underlying cells
    public override void Update()
    {
        foreach ( LineupStarterCell cell in cells )
        {
            cell.Update();
        }
    }

    /* Event Callbacks */

    // User tapped cell for zone 1-6 or liberos
    private void OnCellTapped( LineupPanelCell cell )
	{
		if ( cell is LineupStarterCell starterCell )
		{
			selectedCell = starterCell;

			bool libero = starterCell.IsLibero;
			int zone = starterCell.Zone;

			string prefix = DXString.GetUpper( libero ? "lineup.libero" : "lineup.rot" );
			int zoneAdj = libero ? (zone - Lineup.BaseEntries) : zone;

			// Update popup menu
			LineupMenu.Header = $"{prefix} {zoneAdj}";
			LineupMenu.HasClear = !starterCell.IsEmpty;

			LineupMenu.SetReduced( GetPlayers() );
			LineupMenu.SetDisabled( [ starterCell.Player ] );

			// Display
			LineupMenu.ShowFromView( starterCell );
		}
	}

	// User selected player from lineup menu
	protected override void OnPlayerSelected( Player player, bool cancel )
	{
		selectedCell.SetPlayer( player );

		OnChanged();
	}

	/* Layout */

	// Updates all underlying children
	public override void UpdateLayout( LayoutType type )
	{
        base.UpdateLayout( type );

		// Update children
		foreach ( LineupStarterCell cell in cells )
		{
			cell.UpdateLayout( type );
		}
	}
}

//
