﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.Utils;

namespace iStatVball3;

/*
 * Implements the lineup panel used for both substitutions and libero swaps. Both panel types include 6 cells in a 3x2
 * grid representing zones 1-6.
 */
public class LineupReplace : LineupPanel
{
	/* Properties */
	public bool IsSub => Type == LineupEntryReplace.SubKey;
	public bool IsSwap => Type == LineupEntryReplace.SwapKey;

	// Libero(s) from starting lineup
	public List<Player> Liberos => starters.GetLiberos();

	/* Fields */
	private readonly List<LineupReplaceCell> cells;

	// External ref
	private readonly LineupStarter starters;

	/* Methods */

	// NO UI in constructor
	public LineupReplace( LineupStarter starters, string type )
	{
        this.starters = starters;

		Type = type;

        // Allocate
        cells = new List<LineupReplaceCell>( Lineup.BaseEntries );

		// Create underlying cells without UI
		for ( int zone = 1; zone <= Lineup.BaseEntries; zone++ )
		{
            LineupReplaceCell cell = new( this, zone )
            {
                Changed = OnChanged
            };

            cells.Add( cell );
        }
    }

    // Post construction initialization
    public override void Init( IList<Player> roster, Color color, bool editable )
    {
        base.Init( roster, color, editable );

		IsInitialized = true;
    }

    // Adds individual cell to UI and initializes
    protected override void AddCell( int zone, int col, int row )
    {
		LineupReplaceCell cell = cells[ zone - 1 ];

		// Initialize UI
        cell.Init( Color, IsEditEnabled );

        // Put in correct grid position
        Add( cell, col, row );
    }

    /* Entries */

    // Returns list of all replacement entries from cells 1-6 (DEEP COPY)
    public List<LineupEntryReplace> GetEntries()
	{
		List<LineupEntryReplace> entries = new();

		// Add 0-2 entries from each cell
		foreach ( LineupReplaceCell cell in cells )
		{
			entries.AddRange( cell.GetEntries() );
		}

		return entries;
	}

	// Populates all replacement entries for cells 1-6
	public void SetEntries( IList<LineupEntryReplace> entries )
	{
		if ( entries != null )
		{
			foreach ( LineupEntryReplace entry in entries )
			{
				int index = (entry.Rotation - 1);

				// Populate matching cell
				LineupReplaceCell cell = cells[ index ];
				cell.SetEntry( entry );
			}
		}
	}

	/* Update */

	// Forces redraw of all underlying cells
	public override void Update()
	{
        foreach ( LineupReplaceCell cell in cells )
		{
			cell.Update();
		}
	}

	/* Validation */

	// Only valid if all cells valid, return invalid zone number
	public int Validate()
	{
		foreach ( LineupReplaceCell cell in cells )
		{
			// Invalid
			if ( !cell.IsValid() )
			{
				return cell.Zone;
			}
		}

		// Valid
		return 0;
	}

	// Determines if player subbed/swapped OUT/IN more than once across cells
	public Player FindDuplicate( bool outOnly )
	{
		// Check each player
		foreach ( Player player in Roster )
		{
			int outCount = 0;
			int inCount = 0;

			// Across all cells
			foreach ( LineupReplaceCell cell in cells )
			{
				// Sub/swap OUT
				if ( cell.HasPlayer( player, LineupReplaceRow.RowType.Out ) )
				{
					outCount++;
				}

				// Sub/swap IN
				if ( cell.HasPlayer( player, LineupReplaceRow.RowType.In ) )
				{
					// Libero can be swapped in multiple times
					if ( !outOnly )
					{
						inCount++;
					}
				}

				// Duplicate
				if ( (outCount > 1) || (inCount > 1) )
				{
					return player;
				}
			}
		}

		// No duplicates
		return null;
	}

	/* Event Callbacks */

	// Unused
	protected override void OnPlayerSelected( Player player, bool cancel )
	{}

	/* Layout */

	// Updates all underlying children
	public override void UpdateLayout( LayoutType type )
	{
        base.UpdateLayout( type );

		// Update children
		foreach ( LineupReplaceCell cell in cells )
		{
			cell.UpdateLayout( type );
		}
	}
}

//
