﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI.Gestures;
using DXLib.Utils;

namespace iStatVball3;

/*
 * Displays an individual cell representing one zone in the 1-6 lineup.
 */
public abstract class LineupPanelCell : DXGridGestures
{
	/* Events */
	public Action<LineupPanelCell> CellTapped { get; set; }
	public Action Changed { get; set; }

	/* Properties */

	// 1-6 (can NOT be named Rotation)
	public int Zone { get; private set; }

	// Currently in edit mode?
	public bool IsEditEnabled { get; private set; }

	// Custom component color
	public Color Color { get; private set; }

	// Has cell been initialized?
	public bool IsInitialized { get; protected set; }

	/* Methods */

	// NO UI in constructor
	protected LineupPanelCell( int zone )
	{
		Zone = zone;

		Horizontal = LayoutOptions.Fill;
		Vertical = LayoutOptions.Fill;
	}

    // Post construction initialization
    public virtual void Init( Color color, bool editable )
	{
		Color = color;
		IsEditEnabled = editable;

        // Use down for better performance, mobile scrolls so requires tap
        if ( IsEditEnabled && !IsInitialized )
        {
            if ( DXDevice.IsTablet )
            {
                Down += OnDown;
            }
            else
            {
                Tapped += OnTapped;
            }
        }

		IsInitialized = true;
    }

    // Handles user tap/touch
    protected virtual void HandleTouch( Point point )
	{
		CellTapped?.Invoke( this );
	}

	/* Abstracts */

	// Forces update of all UI based on previously set data values
	public abstract void Update();

	/* Event Callbacks */

	// User touched down anywhere within cell
	private void OnDown( object sender, MR.Gestures.DownUpEventArgs args )
	{
		HandleTouch( args.Center );
	}

	// User tapped anywhere within cell
	private void OnTapped( object sender, MR.Gestures.TapEventArgs args )
	{
		HandleTouch( args.Center );
	}

	/* Layout */

	// Redraws entire cell
	public override void UpdateLayout( LayoutType type )
	{
		double dim = DXDevice.GetScreenDim();
		double margin = (dim * 0.006);

		Margin = new Thickness( (margin - 5), margin, 0, 0 );
	}
}

//
