﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;
using DXLib.UI.Layout;

using DXLib.UI.Control;
using DXLib.UI.Control.Button;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Base class for all lineup panels (starters, subs, swaps). Includes a 3x2 grid of lineup cells for zones 1-6 as well
 * as help text and tooltip.
 */
public abstract class LineupPanel : DXGridLayout
{
	/* Events */
	public Action Changed { get; set; }

    /* Properties */

    // Players
    public IList<Player> Roster { get; private set; }

    // Custom color
    public Color Color { get; private set; }

    // Currently editable?
    public bool IsEditEnabled { get; private set; }

	// Starters, subs, swaps
	public string Type { get; protected set; }

	// Has control been modified?
	public bool HasChanges { get; private set; }

	// Shared throughout panel
	public LineupMenu LineupMenu { get; private set; }

	// Has UI been initialized?
	public bool IsInitialized { get; protected set; }

	/* Fields */
	private DXIconButton helpBtn;

	/* Methods */

	// Post construction initialization
	public virtual void Init( IList<Player> roster, Color color, bool editable )
	{
		Roster = roster;
		Color = color;
		IsEditEnabled = editable;

		BackgroundColor = Color;

		bool tablet = DXDevice.IsTablet;
		bool wide = DXDevice.IsTabletWide;

		double dim = DXDevice.GetScreenDim();

		// Spacing
		Padding = 0;
		RowSpacing = (dim * 0.006);
		ColumnSpacing = (dim * 0.010);

		Horizontal = LayoutOptions.Fill;
		Vertical = LayoutOptions.Fill;

		// 2-3 rows
		AddStarRow();								// 0: frontRow
		AddStarRow();								// 1: backRow
		if ( IsEditEnabled ) AddFixedRow( 30 );		// 2: help

		// 3 columns
		AddStarColumn();	    // 0: left
		AddStarColumn();		// 1: middle
		AddStarColumn();		// 2: right

		// 4 3 2
		// 5 6 1
		AddCell( 1, 2, 1 );
		AddCell( 2, 2, 0 );
		AddCell( 3, 1, 0 );
		AddCell( 4, 0, 0 );
		AddCell( 5, 0, 1 );
		AddCell( 6, 1, 1 );

		if ( IsEditEnabled )
		{
			// Help text/icon
			DXHorizontalLayout helpLayout = new()
			{
				Padding = 0,
				Spacing = 10,
				Margin = new Thickness( 5, 0, 0, (wide ? -10 : 0) ),

				Vertical = LayoutOptions.Center
			};

			DXLabel helpLbl = new()
			{
				Resource = $"lineup.builder.{Type}",

				TextColor = DXColors.Light4,
				Font = DXFonts.Roboto,
				FontSize = tablet ? 17 : 12,
				VAlign = TextAlignment.Center
			};

			helpLayout.Add( helpLbl );

			helpBtn = new DXIconButton()
			{
				Resource = "help",

				IconColor = DXColors.Light4,
				Size = tablet ? 26 : 20,

				IsSticky = true,
				ButtonTapped = OnHelpTapped
			};

			helpLayout.Add( helpBtn );

			Add( helpLayout, 0, 2 );
			SpanColumns( helpLayout, 3 );
		}

		// Reusable menu instance
		if ( Roster != null )
		{
			LineupMenu = new LineupMenu( Roster.Count, true )
			{
				MenuColor = Color,

				PlayerSelected = OnPlayerSelected
			};
		}
	}

    /* Abstracts */

    // Adds individual cell to UI and initializes
    protected abstract void AddCell( int zone, int col, int row );

	// Force UI redraw
	public abstract void Update();

	/* Event Callbacks */

	// User selected player for underlying cell
	protected abstract void OnPlayerSelected( Player player, bool cancel );

	// Control has been modified, save now valid
	protected virtual void OnChanged()
	{
		HasChanges = true;

		// Callback listener
		Changed?.Invoke();
	}

	// Show help tooltip
	private void OnHelpTapped()
	{
		DXTooltip.Show( helpBtn, $"lineup.builder.{Type}" );
	}
}

//
