﻿/* 
 * Copyright (c)2009-2012 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;
using DXLib.UI.Container;
using DXLib.UI.Form.Control;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Implements a custom selector field used to (pre)view a lineup. The lineup will be shown in a non-modal popup.
 */
public class LineupFormControl : DXSelectorField
{
	/* Events */
	public Action PreviewTapped { get; set; }

	/* Properties */
	public Color Color { get; set; }

	// Independently controls disabled state of preview icon
	public bool IsPreviewDisabled {  set => SetPreviewDisabled( value ); }

	/* Methods */

	// Post construction initialization
	public override void Init()
	{
		// Do NOT call base

		HasClear = true;

		AddControl( selector );

		// Override clear button
		clearBtn.Resource = "preview";
		clearBtn.IsSticky = true;

		clearBtn.IconColor = DXColors.Action;
		clearBtn.IconScale = 0.90f;

		UpdateClear();
	}

	// Overrides parent enable/disable
	public override void SetDisabled( bool disabled )
	{
		base.SetDisabled( disabled );

		// Preview button must remain enabled
		IsEnabled = true;

		helpBtn.IsDisabled = disabled;
	}

	// Enables/disables preview eye icon
	private void SetPreviewDisabled( bool disabled )
	{
		clearBtn.IsDisabled = disabled;
	}

	/* Show */

	// Shows lineup zones 1-6 (plus 2 liberos) in non-modal popup
	public void ShowLineup( Lineup lineup )
	{
		bool tablet = DXDevice.IsTablet;

		double dim = DXDevice.GetScreenWd();
		double pad = (dim * 0.014);

		double width = (dim * (tablet ? 0.600: 1.000));
		double height = (dim * (tablet ? 0.450 : 0.400));
		
		// Create popup
		DXPopup popup = new()
		{
			Padding = new Thickness( pad, (pad - 3), pad, pad ),  
			Margin = 0,
			
			Color = Color,
			IsModal = false,
			ShowHeader = false,
			
			ViewWidth = (width + (pad * 2)),
			ViewHeight = (height + (pad * 2)),

			PopupClosed = clearBtn.Reset
		};
		
		// Build lineup data
		LineupStarter starters = new()
		{
			Padding = 0,
			Margin = 0,
			
			Horizontal = LayoutOptions.Fill,
			Vertical = LayoutOptions.Fill
		};

		starters.SetSize( width, height );
		
		starters.Init( null, Color, false );
		starters.SetEntries( lineup.Entries );

		starters.Update();
		starters.UpdateLayout();

		// Display
		popup.SetContent( starters );
		popup.ShowFromView( clearBtn );
	}

	/* Event Callbacks */

	// Control lost focus
	protected override void OnUnfocus()
	{
		base.OnUnfocus();

		// Preview always visible
		clearBtn.IsVisible = true;
	}

	// User tapped preview button
	protected override void OnClearTapped()
	{
		PreviewTapped?.Invoke();
	}
}

//
