﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using Plugin.Firebase.Firestore;

using DXLib.Utils;

namespace iStatVball3;

/* 
 * Represents a lineup replacement for either a substitution or libero swap. Records the unique ID, number, and position
 * for the player coming out as well as the player coming in. Includes additional configuration information required for
 * subs/swaps.
 * 
 * Persisted as a map in the parent Lineup NOT as a root level collection.
 */
public class LineupEntryReplace : IFirestoreObject
{
	/* Constants */
	public const string SubKey = Stats.SubKey;
	public const string SwapKey = Stats.SwapInKey;

    /* Properties */

    // Sub or swap?
    [FirestoreProperty("Type")] public string Type { get; set; }

    // Rotation (1-6) when sub/swap occurs
    [FirestoreProperty("Rotation")] public int Rotation { get; set; }

    // Out/In
    [FirestoreProperty("OutEntry")] public LineupEntry OutEntry { get; set; }
    [FirestoreProperty("InEntry")] public LineupEntry InEntry { get; set; }

    // Require confirmation from user first?
    [FirestoreProperty("Confirm")] public bool Confirm { get; set; }

    // Start or End of possession?
    [FirestoreProperty("Start")] public bool Start { get; set; }

	/* Ignored */
	public bool IsSub => Type == SubKey;
	public bool IsSwap => Type == SwapKey;

	/* Methods */

	// Returns string label representing this replace entry
	public string GetLabel()
	{
		string outName = OutEntry.Player.GetAbbrevNumberName( true );
		string inName = InEntry.Player.GetAbbrevNumberName( true );

		string forStr = DXString.Get( "auto.for" );

		// '#21 J.Smith in for #11 J.Doe'
		return $"{inName} {forStr} {outName}?";
	}

	/* Populate */

	// Populates player FK references (no db)
	public void Populate( Season season )
	{
		OutEntry.Populate( season );
		InEntry.Populate( season );
	}
}

//
