﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI.Card;
using DXLib.Utils;

namespace iStatVball3;

/*
 * Card page for list of Coaches within a parent Season.
 */
public class CoachPage : CardPage
{
	/* Methods */
	public CoachPage( Season season ) : base( season )
	{
		Title = DXString.Get( "coach.plural" );
	}

	/* Abstracts */

	// Displays all coach cards
	public override void LoadCards()
	{
		Clear();

		// Required for permissions
		User user = Shell.CurrentUser;

		if ( data is Season season )
		{
			var coaches = season.Coaches.OrderBy( c => c.Created );

			// Add card for each coach
			foreach ( Coach coach in coaches )
			{
				CardSmall card = new()
				{
					Data = coach,

					DefaultIcon = "coach",
					ImageUrl = coach.ImageUrl,
					ImageColor = season.Color,

					Title = coach.FullName,
					SubTitle = coach.TypeName,

					LinkStatus = coach.Link?.Status,
					HasAnalyze = false,

					IsEditEnabled = Coach.CanEdit( user ),
					IsAnalyzeEnabled = Coach.CanAnalyze( user ),

					IsTappable = false,
					AnalyzeTapped = OnAnalyzeTapped,
					EditTapped = OnEditTapped
				};

				card.Init();
				
				AddCard( card );
			}

			// Add card for creating new objects
			if ( Coach.CanCreate( season, user ) )
			{
				AddNewCard( "coach", CardSmall.CardHt );
			}
		}
	}

	/* Event Callbacks */

	// Called back when 'Add New' tapped
	protected override void OnNewCardTapped( DXCard card )
	{
		Season season = data as Season;
		CoachForm form = new CoachForm( null, season );

		// Open in create mode
		Shell.Instance.ShowForm( form );
	}

	// Called back when card analyze button tapped
	private static void OnAnalyzeTapped( DXCard card )
	{}

	// Called back when card edit button tapped
	private void OnEditTapped( DXCard card )
	{
		Coach coach = card.Data as Coach;
		Season season = data as Season;

		CoachForm form = new( coach, season );

		// Open in edit mode
		Shell.Instance.ShowForm( form );
	}
}

//
