﻿/*
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC.
 */

using DXLib.UI.Form;
using DXLib.UI.Form.Control;

using DXLib.Data;

namespace iStatVball3;

/*
 * CRUD data entry form for a Coach.
 */
public sealed class CoachForm : LinkForm
{
	/* Fields */
	private readonly Season season;

	/* Methods */
	public CoachForm( Coach coach, Season season ) : base( coach, "coach" )
	{
		this.season = season;

		header.Title = HasData ? coach.FullName : CreateHeader();
		imagePanel.Color = season.Color;

		/* Required */

		// Type
		AddControl( new DXSelectorField
		{
			Key = "type",
			Title = "form.type",
			Items = "coach.type",
			SelectedItem = coach?.Type,
			Hint = DXFormControl.HintType.Required,
			Help = "coach.type"
		}, true, true );

		// First Name
		AddControl( new DXTextField
		{
			Key = "first",
			Title = "coach.first",
			Text = coach?.FirstName,
			MinLength = 1,
			MaxLength = 32,
			Type = DXTextField.TextType.CapitalizeWords,
			Hint = DXFormControl.HintType.RequiredMin,
			Help = "coach.first"
		}, true, true );

		// Last Name
		AddControl( new DXTextField
		{
			Key = "last",
			Title = "coach.last",
			Text = coach?.LastName,
			MinLength = 1,
			MaxLength = 32,
			Type = DXTextField.TextType.CapitalizeWords,
			Hint = DXFormControl.HintType.RequiredMin,
			Help = "coach.last"
		}, true, true );

		/* Optional */

		// Nickname
		AddControl( new DXTextField
		{
			Key = "nickname",
			Title = "coach.nickname",
			Text = coach?.Nickname,
			MinLength = 1,
			MaxLength = 32,
			Type = DXTextField.TextType.CapitalizeWords,
			Help = "coach.nickname"
		}, false, false );

		// Notes
		AddControl( new DXEditorField
		{
			Key = "notes",
			Title = "form.notes",
			Text = coach?.Notes,
			MaxLength = 1024,
			Help = "coach.notes"
		}, false, false );

		/* Links */

		// Coach link
		if ( HasData )
		{
			AddControl( new LinkField
			{
				IsDisabled = !HasData || !DXData.HasConnection(),

				Key = "link",
				Title = "coach.link",
				Link = coach?.Link,
				Help = "coach.link"
			});
		}

		// Image
		SetImage( coach?.ImageUrl );

		// Control initialization
		Init();
	}

	/* Event Callbacks */

	// User confirmed delete
	protected override async void OnDeleteConfirmed()
	{
		base.OnDeleteConfirmed();

		if ( data is Coach coach )
		{
			// Cascading delete
			await coach.Delete( true );

			// Refresh UI
			Shell.Instance.HideForm();
		}
	}

	// Used confirmed cancel
	protected override void OnCancelConfirmed()
	{
		base.OnCancelConfirmed();

		Shell.Instance.HideForm();
	}

	// Used tapped save
	protected override async void OnSaveTapped()
	{
		base.OnSaveTapped();

		// Retrieve fields
		string type = GetString( "type", true );
		string firstName = GetString( "first", true );
		string lastName = GetString( "last", true );

		string nickname = GetString( "nickname", false );
		string notes = GetString( "notes", false );

		// Link
		string link = GetLink( "link" );
		
		// Update existing object
		if ( HasData )
		{
			if ( data is Coach coach )
			{
				coach.Type = type;
				coach.FirstName = firstName;
				coach.LastName = lastName;

				coach.Nickname = nickname;
				coach.Notes = notes;

				// Image
				await SaveImage( coach );

				// Update account link
				coach.Link = await UpdateLink( link, coach.Link,
				CreateKeys( Permission.LevelType.Coach, season.Team ) );

				// Persist
				await coach.Update();
			}
		}
		// Create new object
		else
		{
			Coach coach = new Coach
			{
				Type = type,
				FirstName = firstName,
				LastName = lastName,

				Nickname = nickname,
				Notes = notes,

				// Set parent
				SeasonId = season.UniqueId,
				Season = season
			};

			// Image
			await SaveImage( coach );

			// Add account link
			coach.Link = await CreateLink( link, CreateKeys( Permission.LevelType.Coach, season.Team ) );

			// Add to parent
			season.Coaches.Add( coach );

			// Persist
			await coach.Create();
		}

		// Refresh UI
		Shell.Instance.HideForm();
	}
}

//
