﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;
using DXLib.UI.Layout;

using DXLib.UI.Control;
using DXLib.UI.Control.Button;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Implements a header view to be shown at the top of a Play-by-Play log. The header includes optional close and filter
 * buttons as well as the count of displayed log events.
 */
public class LogHeader : DXGridLayout
{
	/* Constants */
	public const double ButtonSize = 26;
	public const double HeaderHt = (ButtonSize + 10);

	/* Events */
	public Action CloseTapped { get; set; }
	public Action FilterTapped { get; set; }

	/* Properties */
	public bool ShowClose { get; set; }
	public bool ShowCount { get; set; }
	public bool ShowFilter { set => SetFilter( value ); }

	public DXIconButton FilterBtn { get; private set; }

	/* Fields */
	private DXIconButton closeBtn;
	private DXLabel countLbl;

	/* Methods */
	public LogHeader()
	{
		BackgroundColor = DXColors.Dark4;
		
		Horizontal = LayoutOptions.Fill;
		Vertical = LayoutOptions.Fill;
		
		IgnoreSafeArea = false;
		IsClippedToBounds = true;

		// Defaults
		ShowClose = false;
		ShowCount = true;
		ShowFilter = false;
	}

	// Post construction initialization
	public void Init()
	{
		bool tablet = DXDevice.IsTablet;

		Padding = new Thickness( 8, 0 );
		RowSpacing = 0;
		ColumnSpacing = 0;

		double buttonColWd = ShowClose ? (tablet ? 50 : 45) : (tablet ? 70 : 65);

		// Close
		AddFixedColumn( buttonColWd );

		if ( ShowClose )
		{
			closeBtn = new DXIconButton
			{
				Resource = "close",
				Size = ButtonSize,
				IconColor = DXColors.Dark1,

				Horizontal = LayoutOptions.Start,
				Vertical = LayoutOptions.Center,

				ButtonTapped = OnCloseTapped
			};
			
			Add( closeBtn, 0, 0 );
		}

		// Count
		if ( ShowCount )
		{
			countLbl = new DXLabel
			{
				Font = DXFonts.Roboto,
				FontSize = 15,

				VAlign = TextAlignment.Center,

				Horizontal = LayoutOptions.Start,
				Vertical = LayoutOptions.Center
			};

			AddStarColumn();
			Add( countLbl, (ColumnCount - 1), 0 );
		}

		// Filter
		AddFixedColumn( buttonColWd );
	}

	// Updates count label with specified value
	public void SetCount( int count, bool filtered )
	{
		countLbl.Text = $"{count} {DXString.Get( "log.events" )}";
		countLbl.TextColor = filtered ? DXColors.Light4 : DXColors.Dark1;
	}

	// Shows/hides optional filter button
	private void SetFilter( bool show )
	{
		// Show
		if ( show )
		{
			// Lazily create
			if ( FilterBtn == null )
			{
				FilterBtn = new DXIconButton
				{
					Resource = "filter",
					Size = ButtonSize,
					IconColor = DXColors.Dark1,

					Margin = 0,
					Horizontal = LayoutOptions.Center,
					Vertical = LayoutOptions.Center,

					IsSticky = true,
					ButtonTapped = OnFilterTapped
				};

				Add( FilterBtn, 2, 0 );
			}

			FilterBtn.IsVisible = true;
		}
		// Hide
		else
		{
			if ( FilterBtn != null )
			{
				FilterBtn.IsVisible = false;
			}
		}
	}

	/* Event Callbacks */

	// User tapped close button, callback listener
	private void OnCloseTapped()
	{
		CloseTapped?.Invoke();
	}

	// User tapped filter button
	private void OnFilterTapped()
	{
		FilterTapped?.Invoke();
	}
}

//
