﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.Email;
using DXLib.Utils;

namespace iStatVball3;

/*
 * Exports an entire Play-by-Play log to PDF and sends as an email attachment.
 */
public static class LogExporter
{
    /* Constants */
    private const string FilePrefix = "istatvball3";
    private const string FileExt = "pdf";
    private const string FileType = "application/pdf";

    /* Methods */

    // Creates PDF document then sends via email
    public static async Task<DXEmail.Result> Export( ExportConfig config )
    {
        LogExport export = new();

        // Create PDF document data
        byte[] pdf = await export.Create( config );

        // Async email send
        DXEmail.Result result = await DXEmail.Send( new DXEmailConfig
        {
            // Local/remote
            IsLocal = config.IsLocal,

            // From:
            FromAddress = config.FromAddress,
            FromName = config.FromName,

            // To:
            ToAddresses = config.ToAddresses,
            ToNames = config.ToNames,

            // Local use only
            Subject = config.Subject,
            Body = DXString.Get( "export.body", config.Title ),

            // Template
            TemplateId = "d-12a8b1a9f59b410098de94bd6f485c3a",
            TemplateData = config,

            // Attachment
            FileRaw = pdf,
            FileName = CreateFilename( config ),
            FileData = DXUtils.ToBase64( pdf ),
            FileType = FileType
        });

        return result;
    }

    // Generates canonical filename for export file
    private static string CreateFilename( ExportConfig config )
    {
        string title = DXString.Get( "log.abbrev" );
        string dateStr = DXUtils.FilenameFromDate( config.Date );

        // 'istatvball3_pbp_team_241025.pdf'
        string fullName = $"{FilePrefix}_{title}_{config.Team}_{dateStr}.{FileExt}";
        string filename = DXUtils.ToFilename( fullName );

        return filename;
    }
}

//
