﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using System.Collections.ObjectModel;

using DXLib.UI;
using DXLib.UI.Layout;
using DXLib.UI.Container;

using DXLib.UI.Control;
using DXLib.UI.Control.Selector;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Implements the interface for selecting a MaxPreps school from a list of all ~30K schools eligible for linking. A
 * search field allows the user to type to filter the full list.
 */
public class MP2Selector : DXPopup
{
    /* Constants */
    private const double IconSize = 30;
    private const double DefaultHt = 50;

    /* Events */
    public Action<MP2School> Selected { get; set; }

    /* Properties */
    public static List<MP2School> SchoolList { get; private set; }
    public MP2School SelectedSchool { get; private set; }

    /* Inherited */

    // PopupClosed
    // ViewWidth
    // ViewHeight

    /* Fields */
    private bool initialized;

    /* Methods */
	public MP2Selector()
	{}

    // Post construction initialization (once only)
    public async Task Init()
    {
        if ( !initialized )
        {
            // Layout
            DXGridLayout layout = new()
            {
                BackgroundColor = DXColors.Light3,

                Padding = 0,
                Margin = 0,

                ColumnSpacing = 0,
                RowSpacing = 0,

                Horizontal = LayoutOptions.Fill,
                Vertical = LayoutOptions.Fill
            };

            // Search icon
            DXIcon icon = new()
            {
                Margin = new Thickness( 5, 5, 0, 0 ),

                Resource = "search",
                Color = DXColors.Dark4,

                Horizontal = LayoutOptions.Fill,
                Vertical = LayoutOptions.Fill
            };

            double viewWd = MP2SchoolView.ViewWd;
            double viewHt = MP2SchoolView.ViewHt;

            double wd = Math.Min( (viewWd + IconSize), DXDevice.GetScreenWd() );
            
            // Create search control
            DXSearchSelector selector = new()
            {
                BackgroundColor = DXColors.Light4,
                Placeholder = DXString.Get( "maxpreps2.search" ),

                Margin = new Thickness( 5, -5, 0, 0 ),
                ItemPadding = new Thickness( 10, 5 ),
                DropDownItemHeight = viewHt,

                WidthRequest = (wd - IconSize),
                HeightRequest = DefaultHt,

                Horizontal = LayoutOptions.Fill,
                Vertical = LayoutOptions.Fill,

                DisplayMemberPath = "Name",
                BindingContext = new MP2School(),

                ItemTemplate = new DataTemplate( () => new MP2SchoolView() ),

                Selected = OnSelected
            };

            // 2 columns, 1 row
            layout.AddFixedColumn( IconSize );         // 0: icon
            layout.AddStarColumn();                    // 1: selector

            layout.Add( icon, 0, 0 );
            layout.Add( selector, 1, 0 );

            ViewWidth = wd;
            ViewHeight = (DefaultHeaderHt + DefaultHt);

            SetTitle( DXString.Get( "maxpreps2.title" ) );
            SetContent( layout );

            // Populate all schools ONCE ONLY
            SchoolList ??= await MP2Exporter.ReadSchools();

            // Attach view to data
            selector.ItemsSource = new ObservableCollection<MP2School>( SchoolList );

            initialized = true;
        }
    }

    /* Event Callbacks */

    // User selected school from list
    private void OnSelected( object data )
    {
        MP2School school = data as MP2School;

        // Save internally only
        SelectedSchool = school;

        // Callback listener
        Selected?.Invoke( school );
    }
}

//
