﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;

using DXLib.UI.Alert;
using DXLib.UI.Layout;
using DXLib.UI.Container;
using DXLib.UI.Control.Button;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Implements the interface for displaying a currently linked MaxPreps school and optionally removing the link to that school.
 */
public class MP2Remover : DXPopup
{
    /* Events */
	public Action Removed { get; set; }

	/* Inherited */

    // PopupClosed
	// ViewWidth
	// ViewHeight

	/* Fields */
	private DXButton removeBtn;
    private bool initialized;

    // Underlying data
    private readonly MP2School school;

    /* Methods */
    public MP2Remover( MP2School school )
	{
		this.school = school;
	}

	// Post construction initialization (once only)
	public void Init()
	{
		if ( !initialized )
		{
			Padding = 0;
			Margin = 0;

			double viewHt = MP2SchoolView.ViewHt;
			
			double wd = Math.Min( 445, DXDevice.GetScreenWd() );
			double ht = (DefaultHeaderHt + viewHt + DXButton.DefaultHt - 20);

			// Layout
			DXGridLayout layout = new()
			{
				BackgroundColor = DXColors.Light4,

				Padding = 0,
				Margin = 0,

				ColumnSpacing = 0,
				RowSpacing = 0,

				WidthRequest = wd,
				HeightRequest = ht,

				Horizontal = LayoutOptions.Fill,
				Vertical = LayoutOptions.Fill
			};

			// Same UI as selector list
			MP2SchoolView schoolView = new( school )
			{
				Padding = 0,
				Margin = new Thickness( 0, 5, 0, 0),

				Horizontal = LayoutOptions.Fill,
				Vertical = LayoutOptions.Center
			};

			// Remove
			removeBtn = new DXButton
			{
				Margin = new Thickness( 0, 5, 0, 0 ),

                Resource = "maxpreps2.remove",
				Type = DXButton.ButtonType.Negative,

				ButtonWd = 110,

				Horizontal = LayoutOptions.Center,
				Vertical = LayoutOptions.Center,

				IsSticky = true,
				ButtonTapped = OnRemoveTapped
			};

			removeBtn.Init();
			
			double schoolHt = (viewHt - 10);

			// 1 column, 2 rows
			layout.AddFixedRow( schoolHt );				// 0: school
			layout.AddStarRow();						// 1: remove

			layout.Fill( DXColors.Light2, 0, 1 );

			layout.Add( schoolView, 0, 0 );
			layout.Add( removeBtn, 0, 1 );

			// Configure popup
            SetTitle( DXString.Get( "maxpreps2.title" ) );
            SetContent( layout );

            ViewWidth = wd;
			ViewHeight = ht;

			initialized = true;
		}
	}

	/* Event Callbacks */

	// User tapped removed, confirm first
	private void OnRemoveTapped( object sender )
	{
		DXAlert.ShowOkCancel( "maxpreps2.title", "maxpreps2.remove.opp", school.Name, OnRemoveConfirmed, () => removeBtn.Reset() );
	}

	// User confirmed removal, callback listener
	private void OnRemoveConfirmed()
	{
        Removed?.Invoke();
    }
}

//
