﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using System.Text;

namespace iStatVball3;

/*
 * Responsible for serialization of match stats into the original MaxPreps text file export format.
 */
public static class MPSerializer
{
	/* Constants */
	private const string MaxPrepsId = "46d48991-eb09-4bd6-b64f-1b4965ca4dd7";
	private const string Headers = "Jersey|MatchGamesPlayed|TotalServes|ServingAces|ServingErrors|ServingPoints|AttacksAttempts|AttacksKills|AttacksErrors|ServingReceivedSuccess|ServingReceivedErrors|BlocksSolo|BlocksAssists|BlocksErrors|BallHandlingAttempt|Assists|AssistsErrors|Digs|DigsErrors";

	/* Methods */

	// Serializes all match stats into MaxPreps format
	public static async Task<string> Serialize( Match match )
	{
		StringBuilder builder = new();

		// Headers
		builder.AppendLine( MaxPrepsId );
		builder.AppendLine( Headers );

		// Configure query
		DataConfig config = new()
		{
			Scope = "match",
			ScopeObject = match,

			Organization = match.Season.Organization,
			IsTeam1 = true,

			Team1 = match.Team1,
			Team2 = null,

			Action = null,
			OuterDim = "player",
			InnerDim = null,

			DataSet = "skills"
		};

		// Aggregate stats within scope
		DataStats stats = await DataFilter.Filter( config );

		// Calculate for each player
		await DataDimension.ProcessDimension( config, stats );

		var metricsList = config.Dimension.Metrics.Values;

		// Add export line for each player
		foreach ( DataMetrics metrics in metricsList )
		{
			builder.AppendLine( SerializePlayer( metrics ) );
		}

		// Do NOT encode here
		return builder.ToString();
	}

	// Serializes individual player match stats into MaxPreps format
	private static string SerializePlayer( DataMetrics metrics )
	{
		if ( metrics is MetricsSkill skills )
		{
			StringBuilder builder = new();

			// Misc
			builder.Append( $"{skills.Player_Number}|" );
			builder.Append( $"{skills.SetsPlayed}|" );

			// Serve
			builder.Append( $"{skills.Serve_Attempts}|" );
			builder.Append( $"{skills.Serve_Aces}|" );
			builder.Append( $"{skills.Serve_Errors}|" );
			builder.Append( $"{skills.Serve_Points}|" );

			// Attack
			builder.Append( $"{skills.Attack_TotalAttempts}|" );
			builder.Append( $"{skills.Attack_TotalKills}|" );
			builder.Append( $"{skills.Attack_TotalErrors}|" );

			int receiveSuccess = (skills.Receive_Attempts - skills.Receive_Errors);

			// Receive
			builder.Append( $"{receiveSuccess}|" );
			builder.Append( $"{skills.Receive_Errors}|" );

			// Block
			builder.Append( $"{skills.Block_Blocks}|" );
			builder.Append( $"{skills.Block_Assists}|" );
			builder.Append( $"{skills.Block_Errors}|" );

			// Set
			builder.Append( $"{skills.Set_Attempts}|" );
			builder.Append( $"{skills.Set_TotalAssists}|" );
			builder.Append( $"{skills.Set_Errors}|" );

			// Defense
			builder.Append( $"{skills.Defense_Digs}|" );
			builder.Append( $"{skills.Defense_Errors}" );

			return builder.ToString();
		}

		return null;
	}
}

//
