﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;
using DXLib.UI.Card;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Defines layout for all large size cards (first column/row) on the Season Home page.
 */
public class HomeCardLarge : HomeCard
{
	/* Constants */
	public const double CardHt = 295;
	
	/* Properties */

	// Passthrough to underlying image grid control
	public CardGrid.GridType Type { set => imageGrid.Type = value; }
	public List<Player> Players { set => imageGrid.SetPlayers( value ); }

	public override Color ImageColor { set { base.ImageColor = value; imageGrid.ImageColor = value; } }

	/* Fields */
	private readonly CardGrid imageGrid;
	
	/* Methods */
	public HomeCardLarge()
	{
		labels.BackgroundColor = DXColors.Light4;

		// Underlying control
		imageGrid = new CardGrid
		{
			DefaultIcon = "player",
			BorderSize = 3,

			IconColor = DXColors.Light4,
			BorderColor = DXColors.Light4
		};

		imageGrid.Init();

		// Widescreen requires separate layout, all others same
		bool wide = DXDevice.IsTabletWide;

		// 2 rows
		layout.AddStarRow( wide ? 60 : 70 );		// 0: image
		layout.AddStarRow( wide ? 40 : 30 );		// 1: everything else

		layout.Add( imageArea, 0, 0 );
		layout.Add( imageGrid, 0, 0 );
		layout.Add( labels, 0, 1 );
	}

	/* Layout */

	// Device/orientation specific layout
	public override void UpdateLayout( LayoutType type )
	{
		const double pad = DXCardPage.BasePadding;
		
		double fullWd = DXDevice.GetScreenWd();
		double fullHt = Shell.ContentHt;

		double width = 0;
		double height = 0;

		const double scale = 0.75;
		
		// Calc image area size based on col,row layout and card size
		switch ( type )
		{
			case LayoutType.Landscape:
			case LayoutType.WideLandscape:
			{
				const double cols = 3;
				const double rows = 2;

				width = (fullWd - (pad * (cols + 1))) / cols;
				height = (fullHt - (pad * (rows + 1))) / rows * 0.70;
				break;
			}
			case LayoutType.Portrait:
			case LayoutType.WidePortrait:
			{
				const double cols = 2;
				const double rows = 4;

				width = (fullWd - (pad * (cols + 1))) / cols;
				height = ((fullHt - (pad * (rows + 1))) * 0.33 * 0.70) + pad;
				break;
			}
			case LayoutType.MobileLandscape:
			{
				double safe = DXDevice.SafeAreaLR().HorizontalThickness;

				width = ((fullWd - safe - (pad * 3)) / 2) - 40;
				height = (CardHt * 0.70);
				break;
			}
			case LayoutType.MobilePortrait:
			{
				width = (fullWd - (pad * 2));
				height = (CardHt * 0.70);
				break;
			}
			default: break;
		}

		imageGrid.SetSize( width, height, scale );
		
		// MUST be last
		imageGrid.UpdateLayout();
	}
}

//
