﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;
using DXLib.UI.Layout;
using DXLib.UI.Control;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Implements header area for the EditLogView. Displays the current set number as well as the team, action, and result
 * of the currently selected event.
 */
public class EditLogHeader : DXGridLayout 
{
	/* Fields */
	private readonly DXLabel setLbl;

	private readonly DXLabel teamHdr;
	private readonly DXLabel teamLbl;

	private readonly DXLabel actionHdr;
	private readonly DXLabel actionLbl;

	private readonly DXLabel resultHdr;
	private readonly DXLabel resultLbl;

	private readonly DXLabel errorHdr;
	private readonly DXLabel errorLbl;

	// Stat display formatting
	private readonly RecordHistory history;

	/* Methods */
	public EditLogHeader()
	{
		bool tablet = DXDevice.IsTablet;

		// Create helper
		history = new RecordHistory
		{
			ActionMode = RecordHistory.ActionModes.Action,

			AbbreviateTeams = !tablet,
			AbbreviateResults = false
		};

		BackgroundColor = DXColors.Light2;

		// Layout
		Padding = new Thickness( (tablet ? 15 : 10), (tablet ? 10 : 4) );
		RowSpacing = 0;
		ColumnSpacing = tablet ? 40 : 25;

		Horizontal = LayoutOptions.Fill;
		Vertical = LayoutOptions.Fill;

		IgnoreSafeArea = true;
		
		// Set number
		setLbl = new DXLabel
		{
			TextColor = DXColors.Dark1,
			Font = DXFonts.RobotoBold,
			FontSize = tablet ? 24 : 20,

			VAlign = TextAlignment.Center,
			Vertical = LayoutOptions.Center
		};

		// Team
		teamHdr = CreateHeader( "team.singular" );
		teamLbl = CreateLabel();

		// Action
		actionHdr = CreateHeader( "history.action" );
		actionLbl = CreateLabel();

		// Result
		resultHdr = CreateHeader( "history.result" );
		resultLbl = CreateLabel();

		// Error
		errorHdr = CreateHeader( "history.error" );
		errorLbl = CreateLabel();

		// Error starts hidden
		errorHdr.IsVisible = false;
		errorLbl.IsVisible = false;

		// 2 rows
		AddStarRow( 50 );		// 0: header
		AddStarRow( 50 );       // 1: label

		// 5 columns
		AddAutoColumn();		// 0: set
		AddAutoColumn();	    // 1: team
		AddAutoColumn();		// 2: action
		AddAutoColumn();		// 3: result
		AddAutoColumn();		// 4: error

		// Add components
		Add( setLbl, 0, 0, 1, 2 );

		Add( teamHdr, 1, 0 );
		Add( teamLbl, 1, 1 );

		Add( actionHdr, 2, 0 );
		Add( actionLbl, 2, 1 );

		Add( resultHdr, 3, 0 );
		Add( resultLbl, 3, 1 );

		Add( errorHdr, 4, 0 );
		Add( errorLbl, 4, 1 );
	}

	// Creates team/action/result header label
	private static DXLabel CreateHeader( string resource )
	{
		return new DXLabel
		{
			IsVisible = false,
			Resource = resource,

			TextColor = DXColors.Dark1,
			Font = DXFonts.RobotoBold,
			FontSize = DXDevice.IsTablet ? 17 : 14,

			VAlign = TextAlignment.Center,
			Vertical = LayoutOptions.Center
		};
	}

	// Creates team/action/result value label
	private static DXLabel CreateLabel()
	{
		return new DXLabel
		{
			TextColor = DXColors.Dark1,
			Font = DXFonts.Roboto,
			FontSize = DXDevice.IsTablet ? 17 : 14,

			LineBreakMode = LineBreakMode.TailTruncation,

			VAlign = TextAlignment.Center,
			Vertical = LayoutOptions.Center
		};
	}

	// Post construction initialization
	public void Init( Set set )
	{
		// Initialize children
		history.Init( set.Match );

		string setStr = DXString.Get( "set.singular" );
		string numStr = set.Number.ToString();

		// 'Set N'
		setLbl.Text = $"{setStr} {numStr}";
	}

	// Populates header with values from specified stat
	public void SetStat( Stat stat )
	{
		teamLbl.Text = history.GetTeam( stat.TeamId );

		actionLbl.Text = history.GetAction( stat );
		resultLbl.Text = history.GetResult( stat );

		bool error = stat.IsError;

		// Error not always shown
		errorHdr.IsVisible = error;
		errorLbl.IsVisible = error;

		if ( error )
		{
			errorLbl.Text = history.GetError( stat );
		}

		// Headers shown
		teamHdr.IsVisible = true;
		actionHdr.IsVisible = true;
		resultHdr.IsVisible = true;
	}

	// Clears all header display fields
	public void ClearStat()
	{
		teamHdr.IsVisible = false;
		actionHdr.IsVisible = false;
		resultHdr.IsVisible = false;
		errorHdr.IsVisible = false;

		teamLbl.Text = null!;
		actionLbl.Text = null!;
		resultLbl.Text = null!;
		errorLbl.Text = null!;
	}
}

//
