﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;
using DXLib.UI.Layout;
using DXLib.UI.Control;
using DXLib.UI.Container;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Implements a view for displaying a TrendGraph legend mapping line series colors to players/rotations.
 */
public class TrendLegend : DXScroll
{
	/* Constants */
	private const double ColorSize = 15;

	/* Fields */
	private readonly DXGridLayout layout;

	// Roster
	private List<Player> players;

	/* Methods */
	public TrendLegend()
	{
		Padding = new Thickness( 15, 15, 15, 0 );
		Orientation = ScrollOrientation.Vertical;

		Horizontal = LayoutOptions.Fill;
		Vertical = LayoutOptions.Fill;

		WidthRequest = 200;
		HeightRequest = 350;

		// Layout
		layout = new DXGridLayout
		{
			Padding = 0,
			RowSpacing = 10,
			ColumnSpacing = 10
		};

		// 2 columns
		layout.AddFixedColumn( ColorSize );		// 0: color
		layout.AddStarColumn();					// 1: object

		Content = layout;
	}

	// Updates color-to-object mapping
	public void UpdateData( DataConfig config, Dictionary<string,List<TrendDatum>> data )
	{
		bool ios = DXDevice.IsIOS;

		Color[] palette = TrendView.DefaultPalette;
		int index = 0;

		// Create row for each color-object (do NOT sort keys)
		foreach ( string key in data.Keys )
		{
			int colorIndex = (index % palette.Length);

			// Color (1px border)
			DXFill colorFill = new()
			{
				Square = (ColorSize - 2),
				Color = palette[ colorIndex ],

				Horizontal = LayoutOptions.Center,
				Vertical = LayoutOptions.Center
			};

			bool team = (key == TrendGraph.TeamKey);

			// Object
			DXLabel objectLbl = new()
			{
				Text = GetLabel( config, key ),
				TextColor = DXColors.Dark1,

				Font = team ? DXFonts.RobotoBold : DXFonts.Roboto,
				FontSize = 14,

				LineBreakMode = LineBreakMode.TailTruncation,
				VAlign = TextAlignment.Center,

				Horizontal = LayoutOptions.Start,
				Vertical = LayoutOptions.Center,

				Margin = DXUtils.Top( ios ? 0 : -4 )
			};

			int row = layout.RowCount;

			// Create row
			layout.AddFixedRow( ColorSize );
			layout.Fill( DXColors.Dark1, 0, row );

			// Add components
			layout.Add( colorFill, 0, row );
			layout.Add( objectLbl, 1, row );

			index++;
		}
	}

	// Returns object label for specified key
	private string GetLabel( DataConfig config, string key )
	{
		// Team 'TEAM'
		if ( key == TrendGraph.TeamKey )
		{
			return DXString.GetUpper( "team.singular" );
		}

		// Object
		switch ( config.OuterDim )
		{
			// Player 'John Doe' (lookup in roster)
			case KeyDimension.PlayerKey:
			{
				// Retrieve roster
				players ??= DataFilter.GetPlayers( config );

				// Lookup in roster (MUST check for root)
				foreach ( Player player in players )
				{
					if ( (player.UniqueId == key) || (player.RootId == key) )
					{
						return player.FullName;
					}
				}

				break;
			}
			// Rotation 'Rotation N'
			case KeyDimension.RotationKey:
			{
				int? rotation = DXUtils.ConvertToInt( key );

				return (rotation == null) ? null : KeyRotation.GetLabel( (int)rotation );
			}
		}

		return null;
	}
}

//
