﻿/*
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC.
 */

using DXLib.UI.Control;
using DXLib.UI.Container;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Report card for the Trend Graph report type. Contains a report header and underlying trend graph.
 */
public class TrendCard : ReportCard
{
	/* Properties */

	// External access required for PDF export
	public TrendView View { get; private set; }

	/* Fields */
	private TrendGraph graph;
	private TrendLegend legend;

	/* Methods */
	public TrendCard( JsonReport json, string dimension ) : base( json, dimension )
	{
		json.Help = DXTooltip.Get( "trend.help", json.Title );
	}

	// Post construction initialization
	public override void Init()
	{
		// Extra button in header
		Options = "list";

		base.Init();

		// Underlying control
		View = new TrendView
		{
			Margin = new Thickness( (Pad - 10), 10, (Pad + 5), Pad )
		};

		layout.Add( View, 1, 1 );

		// Helper class
		graph = new TrendGraph
		{
			View = View
		};

		// Setup chart
		View.Init( json.Trend );
		graph.Init( json.Trend );

		// Pre-create legend
		legend = new TrendLegend();
	}

	// Updates all data displayed on this card
	public override async Task UpdateData( DataConfig config )
	{
		await base.UpdateData( config );

		// Update underlying control			
		await graph.UpdateData( config );

		// Update legend
		legend.UpdateData( config, graph.Data );

		// Required to stop spinner
		OnLoadComplete();
	}

	/* Event Callbacks */

	// User tapped extra header button
	protected override void OnOptionsTapped( View view )
	{
		// Lazily create popup
		popup ??= new DXPopup( "trend.legend", legend )
		{
			AutoSize = true,
			PopupClosed = OnClosed
		};

		// Show
		popup.ShowFromView( view );
	}

	// Reset sticky state
	private void OnClosed()
	{
		header.Reset();
	}

	/* Layout */

	// Redraws entire report card
	public override void UpdateLayout( LayoutType type )
	{
		base.UpdateLayout( type );

		// WORKAROUND: MAUI issues
		switch ( type )
		{
			case LayoutType.MobilePortrait:
			{
				View.Margin = new Thickness( Pad, 10, Pad, Pad );
				break;
			}
			case LayoutType.MobileLandscape:
			{
				double safe = DXDevice.SafeArea().Left;

				View.Margin = new Thickness( (safe + Pad), 10, (safe + Pad), Pad );
				break;
			}
		}

		// Update children
		View.UpdateLayout( type );
	}
}

//
