﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.Log;

namespace iStatVball3;

/*
 * Encapsulates all currently selected keys for each component of a season scoped match filter. Provides methods for
 * filtering a match against the current configuration to determine if it should be included in a stat report.
 */
public class TagFilter
{
	/* Properties */

	// Tag components
	public FilterDimension Matches { get; set; }
	public FilterDimension Opponents { get; set; }
	public FilterDimension Tournaments { get; set; }
	public FilterDimension Types { get; set; }
	public FilterDimension Results { get; set; }
	public FilterDimension Formats { get; set; }
	public FilterDimension Leagues { get; set; }
	public FilterDimension Homes { get; set; }
	public FilterDimension Statisticians { get; set; }
	public FilterDimension Venues { get; set; }

	// Date range
	public DateTimeOffset StartDate { get; set; }
	public DateTimeOffset EndDate { get; set; }

	/* Methods */

	// Filters specified match against all current filter selections
	public bool Filter( Match match )
	{
		DXLog.Debug( "Match:{0} Opp:{1} Tourn:{2} Type:{3} Result:{4} Format:{5} League:{6} Home:{7} Stat:{8} Venue:{9} Date:{10}",
				Matches.Contains( match.UniqueId ),
				Opponents.Contains( match.OpponentId ),
				Tournaments.Contains( match.TournamentId ),
				Types.Contains( match.Type ),
				Results.Contains( match.Result ),
				Formats.Contains( match.ScoreFormat ),
				Leagues.Contains( match.League ),
				Homes.Contains( match.HomeAway ),
				Statisticians.Contains( match.StatisticianId ),
				Venues.Contains( match.VenueId ),
				FilterDate( match ) );

		return // All tags
			   Matches.Contains( match.UniqueId ) &&
			   Opponents.Contains( match.OpponentId ) &&
			   Tournaments.Contains( match.TournamentId ) &&
			   Types.Contains( match.Type ) &&
			   Results.Contains( match.Result ) &&
			   Formats.Contains( match.ScoreFormat ) &&
			   Leagues.Contains( match.League ) &&
			   Homes.Contains( match.HomeAway ) &&
			   Statisticians.Contains( match.StatisticianId ) &&
			   Venues.Contains( match.VenueId ) &&

			   // Date range
			   FilterDate( match );
	}

	// Filters specified match against currently selected date range
	private bool FilterDate( Match match )
	{
		// Must be in-progress or ended
		if ( !match.IsNew )
		{
			DateTimeOffset date = match.MatchTime;

			// Pad last day to make sure included
			DateTimeOffset endDate = EndDate.AddDays( 1 );

			// Must be within date range
			bool afterStart = (date.CompareTo( StartDate ) >= 0);
			bool beforeEnd = (date.CompareTo( endDate ) <= 0);

			return afterStart && beforeEnd;
		}

		return false;
	}
}

//
