﻿/*
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC.
 */

using SkiaSharp;
using SkiaSharp.Views.Maui;

using DXLib.UI;
using DXLib.UI.Alert;
using DXLib.UI.Layout;
using DXLib.UI.Container;

using DXLib.UI.Control;
using DXLib.UI.Control.Button;
using DXLib.UI.Control.Checkbox;

using DXLib.Data;
using DXLib.Utils;

namespace iStatVball3;

/*
 * Content area for popup menu used to adjust options for a shot chart. Options include turning legend items on/off as
 * well as turning shot marker components on/off.
 */
public class ShotOptions : DXScroll
{
	/* Events */
	public Action Confirmed;

	/* Properties */
	public DXCheckboxList Legend { get; private set; }
	public DXCheckboxList Options { get; private set; }

	public string ShotType { get; set; }
	public string Success { get; set; }

	// External ref
	public ShotCard Card { get; set; }

	/* Fields */
	private DXVerticalLayout layout;

	/* Methods */
	public void Init()
	{
		bool ios = DXDevice.IsIOS;

		Orientation = ScrollOrientation.Vertical;

		// Underlying layout
		layout = new DXVerticalLayout
		{
			Padding = new Thickness( 17, 15, 17, 10 ),
			Spacing = (ios ? 10 : 4)
		};

		// Legend title
		DXLabel legendLbl = new() 
		{
			Resource = "shot.legend",
			TextColor = DXColors.Dark1,
			Font = DXFonts.RobotoBold,
			FontSize = 17
		};

		layout.Add( legendLbl );

		const double indent = 15;
		const double spacer = 8;

		// Legend checkboxes
		Legend = new DXCheckboxList
		{
			Margin = DXUtils.Left( indent ),
			IsTriState = false,
		};

		switch ( ShotType )
		{
			// Result (attempt/success/error)
			case ShotChart.ResultType:
			{
				Legend.LoadItems( "shot.result" );
				Legend.SelectItems( new List<string> { "att", "suc", "err" } );

				SetColor( "att", ShotView.AttemptColors[0] );
				SetColor( "suc", ShotView.SuccessColors[0] );
				SetColor( "err", ShotView.ErrorColors[0] );

				Legend.SetText( "suc", Success );
				break;
			}
			// Rating (0-3 or 0-4)
			case ShotChart.RatingType:
			{
				int count = Card.RatingCount;

				List<DXItem> items = new List<DXItem>( count );
				List<string> selected = new List<string>( count );

				string ratingStr = DXString.Get( "shot.rating" );

				// Dynamically build rating list
				for ( int r = 0; r < count; r++ )
				{
					string rStr = r.ToString();

					// 'Rating X'
					items.Add( new DXItem( rStr, $"{ratingStr} {r}" ) );
					selected.Add( rStr );
				}

				Legend.LoadItems( items );
				Legend.SelectItems( selected );

				// Colors must be set last
				for ( int r = 0; r < count; r++ )
				{
					SetColor( r.ToString(), ShotView.RatingColors[0][r] );
				}

				break;
			}
		}

		layout.Add( Legend );

		// Options title
		DXLabel optionsLbl = new DXLabel
		{
			Margin = DXUtils.Top( spacer ),

			Resource = "shot.marker",
			TextColor = DXColors.Dark1,
			Font = DXFonts.RobotoBold,
			FontSize = 17
		};

		layout.Add( optionsLbl );

		// Options checkboxes
		Options = new DXCheckboxList
		{
			Margin = DXUtils.Left( indent ),
			IsTriState = false
		};

		Options.LoadItems( "shot.marker" );
		Options.SelectItems( new List<string> { "start", "arrow", "end" } );
		Options.SetColor( DXColors.Dark4 );

		layout.Add( Options );

		// OK
		DXButton okBtn = new()
		{
			Resource = "alert.ok",
			Type = DXButton.ButtonType.Action,

			Margin = DXUtils.Top( spacer ),
			Vertical = LayoutOptions.End,

			ButtonTapped = OnOkTapped
		};

		okBtn.Init();
		
		layout.Add( okBtn );

		WidthRequest = 200;
		HeightRequest = (ShotType == ShotChart.ResultType) ? 380 : ((Card.RatingCount == 4) ? 420 : 455);

		Content = layout;
	}

	// Sets color of specified item
	private void SetColor( string key, SKColor color )
	{
		Legend.SetColor( key, color.ToMauiColor() );
	}

	/* Event Callbacks */

	// User confirmed, callback listener
	private void OnOkTapped( object sender )
	{
		// Must have at least one item selected
		if ( Legend.SelectedCount() > 0 && Options.SelectedCount() > 0 )
		{
			Confirmed?.Invoke();
		}
		else
		{
			DXAlert.ShowOk( "shot.err", "shot.err.msg" );
		}
	}
}

//
