﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

namespace iStatVball3;

/*
 * Helper class for managing data between the shot card and shot view.
 */
public class ShotChart
{
	/* Constants */

	// Chart modes 
	public const string ResultType = "result";
	public const string RatingType = "rating";

	/* Properties */

	// Toggle marker components
	public bool ShowAttempt { get; set; }
	public bool ShowSuccess { get; set; }
	public bool ShowError { get; set; }

	public bool[] ShowRating { get; set; }

	// External refs
	public ShotCard Card { get; set; }
	public ShotView View { get; set; }

	/* Fields */

	// Underlying data
	private DataStats data;

	// External configuration
	private JsonShot jsonShot;

	/* Methods */

	// Post construction initialization
	public void Init( JsonShot json )
	{
		jsonShot = json;

		// Set defaults depending on mode
		switch ( json.Type )
		{
			// Result (att-succ-err)
			case ResultType:
			{
				ShowAttempt = true;
				ShowSuccess = true;
				ShowError = true;
				break;
			}
			// Rating (0-3 or 0-4)
			case RatingType:
			{
				int count = Card.RatingCount;
				ShowRating = new bool[ count ];

				for ( int i = 0; i < count; i++ )
				{
					ShowRating[i] = true;
				}

				break;
			}
		}
	}

	// Updates list of stats to be drawn by this chart
	public void UpdateData( DataConfig config, DataStats stats )
	{
		View.IsTeam1 = config.IsTeam1;

		// Check for permission access, filter dimension(s)
		data = config.Filter( stats );

		Draw();
	}

	// Filters specified result based on whether result type enabled
	private bool IsResultOn( string result )
	{
		return result switch
		{
			Stats.AttemptKey => ShowAttempt,
			Stats.ErrorKey => ShowError,
			Stats.AceKey or 
			Stats.KillKey or 
			Stats.AssistKey => ShowSuccess,
			
			_ => false
		};
	}

	// Filters specified rating based on whether rating is enabled
	private bool IsRatingOn( int? rating )
	{
		int mapped = Stat.MapRating( rating, Card.IsPassing() );

		// Map between 0-4 and 0-3
		if ( (mapped >= 0) && (mapped < Card.RatingCount) )
		{
			return ShowRating[ mapped ];
		}

		return false;
	}

	// Filters then draws shot list
	public void Draw()
	{
		List<Stat> shots = FilterShots();

		// Draw
		View.DrawShots( shots );
	}

	// Filters stats based on result/rating options
	public List<Stat> FilterShots()
	{
		List<Stat> shots = [];

		// Iterate data set
		foreach ( Stat stat in data )
		{
			// Options
			bool showing = jsonShot.Type switch
			{
				ResultType => IsResultOn( stat.Result ),
				RatingType => IsRatingOn( stat.Rating ),
				
				_ => false
			};

			// Add to shot list
			if ( showing )
			{
				shots.Add( stat );
			}
		}

		return shots;
	}
}

//
