﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI.Container;
using DXLib.UI.Control.Checkbox;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Report card for the shot chart report type. Contains a report header and shot chart. Adds a settings button to the header.
 */
public class ShotCard : ReportCard
{
	/* Properties */
	public ShotChart Chart { get; private set; }
	public ShotView View { get; private set; }

	// Chart drawn from Team 1 perspective?
	public bool IsTeam1 { get; private set; }

	// Results or ratings?
	public string ShotType => json.Shot.Type;

	/* Fields */

	// Options popup
	private ShotOptions options;

	/* Methods */
	public ShotCard( JsonReport json, string dimension ) : base( json, dimension )
	{}

	// Post construction initialization
	public override void Init()
	{
		// Extra button in header
		Options = "settings";

		base.Init();

		// Underlying component
		View = new ShotView
		{
			Margin = new Thickness( (Pad - 10), 10, (Pad - 5), (Pad + 10) ),
			
			Card = this,
			LoadComplete = OnLoadComplete
		};
		
		// Helper class
		Chart = new ShotChart
		{
			Card = this,
			View = View
		};

		// Default to Team 1
		View.ShotType = ShotType;
		View.IsTeam1 = true;

		View.Init();

		// Setup chart
		Chart.Init( json.Shot );
		
		// Add to layout
		layout.Add( View, 1, 1 );
	}

	// Updates all data displayed on this card
	public override async Task UpdateData( DataConfig config )
	{
		IsTeam1 = config.IsTeam1;

		await base.UpdateData( config );

		// Update underlying control			
		Chart.UpdateData( config, stats );
	}

	/* Event Callbacks */

	// User tapped extra header button
	protected override void OnOptionsTapped( View view )
	{
		// Lazily create popup
		if ( popup == null )
		{
			JsonShot config = json.Shot;

			// Configure
			options = new ShotOptions
			{
				ShotType = ShotType,
				Success = config.Success,
				Card = this,

				Confirmed = OnConfirmed
			};

			options.Init();

			// Create popup
			popup = new DXPopup( "report.options", options )
			{
				AutoSize = true,
				PopupClosed = OnClosed
			};
		}

		// Show
		popup.ShowFromView( view );
	}

	// User tapped OK
	private void OnConfirmed()
	{
		DXCheckboxList legend = options.Legend;

		// Read legend options
		switch ( ShotType )
		{
			// Result
			case ShotChart.ResultType:
			{
				Chart.ShowAttempt = legend.IsSelected( "att" );
				Chart.ShowSuccess = legend.IsSelected( "suc" );
				Chart.ShowError = legend.IsSelected( "err" );
				break;
			}
			// Rating
			case ShotChart.RatingType:
			{
				int count = Math.Min( RatingCount, Chart.ShowRating.Length );

				for ( int i = 0; i < count; i++ )
				{
					Chart.ShowRating[i] = legend.IsSelected( i.ToString() );
				}

				break;
			}
		}

		DXCheckboxList opt = options.Options;

		// Read marker options
		View.ShowStart = opt.IsSelected( "start" );
		View.ShowArrow = opt.IsSelected( "arrow" );
		View.ShowEnd = opt.IsSelected( "end" );

		// Update chart
		Chart.Draw();

		popup.Hide();
	}

	// Reset sticky state
	private void OnClosed()
	{
		header.Reset();
	}

	/* Layout */

	// Device/orientation specific layout
	public override void UpdateLayout( LayoutType type )
	{
		base.UpdateLayout( type );

		// WORKAROUND: MAUI issues
		switch ( type )
		{
			case LayoutType.MobilePortrait:
			{
				View.Margin = new Thickness( Pad, 10, Pad, Pad );
				break;
			}
			case LayoutType.MobileLandscape:
			{
				double safe = DXDevice.SafeArea().Left;

				View.Margin = new Thickness( (safe + Pad), 10, (safe + Pad), Pad );
				break;
			}
		}

		// Update children
		View.UpdateLayout( type );
	}
}

//
