﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;
using DXLib.UI.Layout;
using DXLib.UI.Control;
using DXLib.UI.Container;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Responsible for laying out all report cards on a report form. The card list is vertically scrollable. The layout can
 * combine reports of any type and data set.
 */
public class ReportLayout : DXScroll
{
	/* Constants */
	public const double BasePadding = 15;

	/* Properties */

	// Grid, bar, custom, etc
	public string Type { get; set; }

	// Stack of report cards
	public List<ReportCard> Cards { get; }

	/* Fields */
	private readonly DXVerticalLayout layout;

	// State control
	private int loadedCount;

	/* Methods */
	public ReportLayout()
	{
		BackgroundColor = DXColors.Dark1;
		Padding = new Thickness( 0, 0, 0, 15 );

		Orientation = ScrollOrientation.Vertical;

		// Cards arranged in vertical list
		layout = new DXVerticalLayout
		{
			BackgroundColor = DXColors.Dark1,

			Padding = 0,
			Spacing = 15,
			
			// Do NOT fill vertically
			Vertical = LayoutOptions.Start,
			
			Horizontal = LayoutOptions.Fill,
			VerticalOptions = LayoutOptions.Fill,
			
			IgnoreSafeArea = true
		};

		Content = layout;

		// Allocate container
		Cards = [];
	}

	// Creates card for all reports of configured type
	public void Init( JsonDashboard json, DataConfig config )
	{
		Cards.Clear();
		layout.Clear();

		// Create each report
		for ( int index = 0; index < json.Reports.Count; index++ )
		{
			JsonReport jsonReport = json.Reports[ index ];

			// CUSTOM ONLY: Get pre-configured dimension for report
			string dimension = json.IsCustom ? json.CustomDimensions[ index ] : null;

			// Cards can be hidden during debug or excluded for dimension(s)
			if ( (jsonReport != null) && jsonReport.IsValid( config ) && ((Type == AnalyzeKeys.CustomKey) || (jsonReport.Type == Type)) )
			{
				ReportCard card = null;

				switch ( jsonReport.Type )
				{
					// Grid
					case AnalyzeKeys.GridKey:
					{
						card = new GridCard( jsonReport, dimension );
						break;
					}
					// Bar
					case AnalyzeKeys.BarKey:
					{
						card = new BarCard( jsonReport, dimension );
						break;
					}
					// Shot
					case AnalyzeKeys.ShotKey:
					{
						card = new ShotCard( jsonReport, dimension );
						break;
					}
					// Heat
					case AnalyzeKeys.HeatKey:
					{
						card = new HeatCard( jsonReport, dimension );
						break;
					}
					// Trend
					case AnalyzeKeys.TrendKey:
					{
						card = new TrendCard( jsonReport, dimension );
						break;
					}
					// Momentum
					case AnalyzeKeys.MomentumKey:
					{
						card = new MomentumCard( jsonReport, dimension );
						break;
					}
				}

				if ( card != null )
				{
					// Register for events
					card.LoadComplete = OnLoadComplete;

					card.Init();

					// Add to layout
					Cards.Add( card );
					layout.Add( card );
				}
			}
		}

		UpdateLayout();
	}

	// Refreshes data display for all cards in layout
	public async Task UpdateData( DataConfig config )
	{
		foreach ( ReportCard card in Cards )
		{
			// MUST create new config instance for each card
			DataConfig newConfig = new( config );

			// Update
			await card.UpdateData( newConfig );
		}
	}

	/* Event Callbacks */

	// One card on layout was fully loaded
	private async void OnLoadComplete()
	{
		loadedCount++;

		// Stop progress spinner once all cards loaded
		if ( loadedCount == Cards.Count )
		{
			DXSpinner.Stop();

			await ScrollToTopWait();
		}
	}

	/* Layout */

	// Device/orientation specific layout
	public override void UpdateLayout( LayoutType type )
	{
		// Account for mobile notch
		Padding = DXUtils.AddPadding( new Thickness( BasePadding ), DXDevice.SafeAreaLR() );

		// Mobile portrait scrolls horizontally
		Orientation = (type == LayoutType.MobilePortrait) ? ScrollOrientation.Both : ScrollOrientation.Vertical;

		// Update child cards
		foreach ( ReportCard card in Cards )
		{
			card.UpdateLayout( type );
		}
	}
}

//
