﻿/*
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC.
 */

using DXLib.UI;
using DXLib.UI.Layout;

using DXLib.UI.Control;
using DXLib.UI.Control.Button;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Displays a header title tooltip button for a report card. An options button can also be displayed when needed.
 */  
public class ReportHeader : DXGridLayout
{
	/* Events */
	public Action<View> OptionsTapped { get; set; }

	/* Properties */

	// Grid title loaded from JSON
	public string Title => titleLbl.Text;

	// Custom options button resource (null if NA)
	public string Options { get; set; }

	/* Fields */
	private DXLabel titleLbl;

	private DXIconButton optionsBtn;
	private DXIconButton helpBtn;

	// Config
	private JsonReport reportJson;

	/* Methods */

	// Post construction initialization
	public void Init( JsonReport json )
	{
		reportJson = json;

		bool tablet = DXDevice.IsTablet;
		bool large = DXDevice.IsLargeTablet();

		double pad = ReportCard.Pad;
		double halfPad = (pad / 2);

		// Must use margin here rather than parent padding
		Margin = new Thickness( pad, halfPad, (pad - 5), 0 );
		ColumnSpacing = 5;

		Horizontal = LayoutOptions.Fill;
		Vertical = LayoutOptions.Fill;
		
		IgnoreSafeArea = true;
		
		// Title
		titleLbl = new DXLabel
		{
			Text = json.Title,

			TextColor = DXColors.Dark1,
			Font = DXFonts.RobotoBold,
			FontSize = tablet ? (large ? 22 : 20) : 17,

			VerticalTextAlignment = TextAlignment.Center,
			Vertical = LayoutOptions.Center
		};

		AddStarColumn();
		Add( titleLbl, 0, 0 );

		const double iconSize = 25;
		bool hasOptions = (Options != null);

		// Options
		if ( hasOptions )
		{
			optionsBtn = new DXIconButton
			{
				Resource = Options,
				IconColor = DXColors.Dark4,
				Size = iconSize,

				IsSticky = true,
				ButtonTapped = OnOptionsTapped,

				Horizontal = LayoutOptions.Center,
				Vertical = LayoutOptions.Center
			};

			AddFixedColumn( iconSize + 5 );
			Add( optionsBtn, 1, 0 );
		}

		// Tooltip
		helpBtn = new DXIconButton
		{
			IsVisible = (json.Help != null),

			Resource = "help",
			IconColor = DXColors.Dark4,
			Size = iconSize,

			IsSticky = true,
			ButtonTapped = OnHelpTapped,

			Horizontal = LayoutOptions.Center,
			Vertical = LayoutOptions.Center
		};

		AddFixedColumn( iconSize + 5 );
		Add( helpBtn, (hasOptions ? 2 : 1), 0 );
	}

	// Reset sticky state
	public void Reset()
	{
		optionsBtn.Reset();
	}

	/* Event Callbacks */

	// Shows optional options popup for report
	private void OnOptionsTapped()
	{
		OptionsTapped?.Invoke( optionsBtn );
	}

	// Shows tooltip popup for report
	private void OnHelpTapped()
	{
		string help = reportJson.Help;

		if ( !string.IsNullOrEmpty( help ) )
		{
			DXTooltip.ShowText( helpBtn, help );
		}
	}
}

//
