﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI.Card;
using DXLib.UI.Control;
using DXLib.UI.Container;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Parent class for all types of report card (grid, shot chart, etc). Each card contains a title header followed by the
 * report specific component. 
 */ 
public abstract class ReportCard : DXCard
{
	/* Constants */
	public static readonly double Pad = DXDevice.IsTablet ? 30 : 15;
	public static readonly double HeaderHt = DXDevice.IsIOS ? 40 : 47;

	// Left-hand vertical stripe
	public const double StripeWd = 15;

	/* Events */
	public Action LoadComplete { get; set; }

	/* Properties */

	// Report specific config
	public string Type => json.Type;
	public string DataSet => json.Data;
	public string Action => json.Action;

	// Grid title loaded from JSON
	public string Title => header.Title;

	// Custom header options button resource (null if NA)
	public string Options { get; set; }

	// 0-3 or 0-4 ratings
	public int RatingCount => (IsPassing() && Shell.Settings.IsPass03) ? 4 : 5;

	/* Fields */

	// External config
	protected readonly JsonReport json;

	// CUSTOM ONLY: Stores pre-configured dimension
	private readonly string dimension;

	// UI controls
	protected ReportHeader header;
	private DXFill stripe;

	// Options popup
	protected DXPopup popup;

	// Filtered stats used by all concrete report types
	protected DataStats stats;

	/* Methods */
	protected ReportCard( JsonReport json, string dimension )
	{
		this.json = json;
		this.dimension = dimension;
	}

	// Post-construction initialization
	public override void Init()
	{
		base.Init();
		
		IsTappable = false;

		double halfPad = (Pad / 2);

		// Children must use margin rather than padding here (color bar)
		Padding = 0;

		// REQUIRED
		layout.IgnoreSafeArea = true;
		
		layout.RowSpacing = halfPad;
		layout.ColumnSpacing = 0;

		// Team color bar
		stripe = new DXFill();

		// Header
		header = new ReportHeader
		{
			Options = Options,
			OptionsTapped = OnOptionsTapped
		};

		header.Init( json );

		// 2 columns
		layout.AddFixedColumn( StripeWd );			// 0: stripe
		layout.AddStarColumn();						// 1: content
		
		// 2 rows
		layout.AddFixedRow( HeaderHt );				// 0: header
		layout.AddAutoRow();						// 1: content

		// Add components (content added by child)
		layout.Add( stripe, 0, 0, 1, 2 );
		layout.Add( header, 1, 0 );
	}

	// Reporting on a passing action?
	public bool IsPassing()
	{
		return Stat.IsPassing( json.Action );
	}

	// Updates data common to all report card types
	public virtual async Task UpdateData( DataConfig config )
	{
		// Inject report specific fields
		config.Type = Type;
		config.DataSet = DataSet;
		config.Action = Action;

		// CUSTOM ONLY: Use pre-configured dimension
		if ( dimension != null )
		{
			config.OuterDim = dimension;
		}

		// Filter by scope, etc
		stats = await DataFilter.Filter( config );

		// Custom color
		stripe.Color = config.IsTeam1 ? config.Organization.Color : ((config.Team2 == null) ? Opponent.DefaultColor : config.Team2.Color);
	}

	/* Event Callbacks */

	// Not all card types have options
	protected virtual void OnOptionsTapped( View view )
	{}

	// Concrete card type was fully loaded, notify listener
	protected void OnLoadComplete()
	{
		LoadComplete?.Invoke();
	}

	/* Layout */

	// Device/orientation specific layout
	public override void UpdateLayout( LayoutType type )
	{
		double stripeWd = DXDevice.IsMobile ? ((type == LayoutType.MobilePortrait) ? 10 : 15) : (DXDevice.GetScreenWd() * 0.015);

		// Stripe narrower on mobile portrait
		layout.ColumnDefinitions[0].Width = stripeWd;
	}
}

//
