﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using Syncfusion.Maui.Charts;

using DXLib.UI;
using DXLib.UI.Layout;
using DXLib.UI.Control;
using DXLib.UI.Container;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Represents an individual annotation dot on a Momentum Chart line series.
 */
public class MomentumDot : EllipseAnnotation
{
	/* Constants */

	// Available annotation types
	public enum DotType
	{
		Timeout,
		Substitution
	};

	// Circle radius
	public const float DotSize = 15;

	// Colors
	private static readonly Color TimeoutColorPos = DXColors.Positive;
	private static readonly Color TimeoutColorNeg = DXColors.Negative;

	private static readonly Color SubColor = DXColors.Accent4;

	/* Properties */
	public DotType Type { get; set; }

	// Graph coordinates
	public int X { set => X1 = value; }
	public int Y { set => Y1 = value; }

	// X-axis scores
	public int ScoreA { get; set; }
	public int ScoreB { get; set; }

	// Underlying data
	public Stat Stat { get; set; }

	// External ref
	public Match Match { get; set; }
	
	/* Methods */
	public MomentumDot()
	{
		IsVisible = true;
		CoordinateUnit = ChartCoordinateUnit.Axis;

		// Circle
		Width = DotSize;
		Height = DotSize;

		// Border
		Stroke = DXColors.Dark1;
		StrokeWidth = 1;
	}

	// Post construction initialization
	public void Init( bool team1 )
	{
		switch ( Type )
		{
			// Timeout (color based on team)
			case DotType.Timeout:
			{
				bool statTeam1 = Match.IsTeam1( Stat.TeamId );
				bool sameTeam = (team1 && statTeam1) || (!team1 && !statTeam1);

				Fill = sameTeam ? TimeoutColorNeg : TimeoutColorPos;
				break;
			}
			// Substitution (Team 1 only)
			case DotType.Substitution:
			{
				Fill = SubColor;
				break;
			}
			default: break;
		}
	}

	/* Popup */

	// Displays detailed annotation info in popup when dot tapped
	public static void ShowPopup( MomentumDot dot, double x, double y )
	{
		bool ios = DXDevice.IsIOS;
		bool sub = (dot.Type == DotType.Substitution);

		// Layout
		DXVerticalLayout layout = new()
		{
			BackgroundColor = DXColors.Light4,

			Padding = new Thickness( 10, 5 ),
			Spacing = ios ? 5 : 0,

			WidthRequest = sub ? 165 : 125,
			HeightRequest = 95
		};

		// Title
		DXLabel titleLbl = new()
		{
			Text = GetTitle( dot ),
			TextColor = DXColors.Dark1,

			HAlign = TextAlignment.Center,

			Font = DXFonts.RobotoBold,
			FontSize = 14
		};

		layout.Add( titleLbl );

		// Line 1
		DXLabel line1Lbl = new()
		{
			Text = GetLine1( dot ),
			TextColor = DXColors.Dark3,

			HAlign = TextAlignment.Center,
			LineBreakMode = LineBreakMode.TailTruncation,

			Font = DXFonts.Roboto,
			FontSize = 14
		};

		layout.Add( line1Lbl );

		// Line 2
		DXLabel line2Lbl = new()
		{
			Text = GetLine2( dot ),
			TextColor = DXColors.Dark3,

			HAlign = TextAlignment.Center,
			LineBreakMode = LineBreakMode.TailTruncation,

			Font = DXFonts.Roboto,
			FontSize = 14
		};

		layout.Add( line2Lbl );

		// Score
		DXLabel scoreLbl = new()
		{
			Text = GetScore( dot ),
			TextColor = DXColors.Dark3,

			HAlign = TextAlignment.Center,

			Font = DXFonts.OswaldBold,
			FontSize = 14
		};

		layout.Add( scoreLbl );

		// Display popup
		DXPopup popup = new( layout );
		popup.Show( x, y );
	}

	// Returns popup title based on annotation type
	private static string GetTitle( MomentumDot dot )
	{
		return dot.Type switch
		{
			DotType.Timeout => DXString.Get( "momentum.timeout" ),
			DotType.Substitution => DXString.Get( "momentum.sub" ),

			_ => null
		};
	}

	// Returns first annotation specific data line
	private static string GetLine1( MomentumDot dot )
	{
		Match match = dot.Match;
		Stat stat = dot.Stat;

		switch ( dot.Type )
		{
			// Timeout (team name)
			case DotType.Timeout:
			{
				return match.IsTeam1( stat.TeamId ) ? match.Team1Name : match.Team2Name;
			}
			// Substitution (player subbing out)
			case DotType.Substitution:
			{
				return $"{stat.Player.GetNumberName( true )} ({DXString.Get( "momentum.sub.out" )})";
			}
			default: break;
		}

		return null;
	}

	// Returns second annotation specific data line
	private static string GetLine2( MomentumDot dot )
	{
		Stat stat = dot.Stat;

		return dot.Type switch
		{
			// Timeout (set timestamp)
			DotType.Timeout => DXUtils.FromDurationShort( stat.Offset ),
			
			// Substitution (player subbing in)
			DotType.Substitution => $"{stat.Player2.GetNumberName( true )} ({DXString.Get( "momentum.sub.in" )})",
			
			_ => null
		};
	}

	// Returns formatted score at time of annotation
	private static string GetScore( MomentumDot dot )
	{
		return $"{dot.ScoreA}-{dot.ScoreB}";
	}
}

//
