﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

namespace iStatVball3;

/*
 * Helper class responsible for generating data to populate a Momentum Chart.
 */
public class MomentumChart
{
	/* Properties */

	// External ref
	public MomentumView View { get; set; }

	/* Methods */

	// Generates entire Momentum Chart data series
	public void UpdateData( DataConfig config, DataStats stats )
	{
		bool team1 = config.IsTeam1;

		// Series data
		PlotMomentum( team1, stats );

		// Annotation markers
		PlotAnnotations( team1, stats );
	}

	/* Line Series */

	// Plots momentum line series data for specified set stats
	private void PlotMomentum( bool team1, DataStats stats )
	{
		// Start with '0-0'
		List<MomentumDatum> data = [ new ( FormatScore( 0, 0 ), 0 ) ];

		// Process each stat
		foreach ( Stat stat in stats )
		{
			if ( stat.IsPoint )
			{
 				MomentumDatum datum = PlotScore( team1, stat );
				data.Add( datum );
			}
		}

		// Render series
		View.LoadData( [ ..data ] );
	}

	// Plots individual score data point on momentum line series
	private static MomentumDatum PlotScore( bool team1, Stat stat )
	{
		StatPoint point = stat.Point;

		// Team scores
		int scoreA = team1 ? point.Score1 : point.Score2;
		int scoreB = team1 ? point.Score2 : point.Score1;

		// X,Y value
		string score = FormatScore( scoreA, scoreB );
		int momentum = (scoreA - scoreB);

		return new MomentumDatum( score, momentum );
	}

	// Formats X-axis score label
	private static string FormatScore( int scoreA, int scoreB )
	{
		// Attempt to horizontally center
		string strA = scoreA.ToString().PadLeft( 2 );
		string strB = scoreB.ToString().PadLeft( 2 );

		// '15'
		// '10'
		return $"{strA}\n{strB}";
	}

	/* Annotations */

	// Adds all annotation dots to line series
	private void PlotAnnotations( bool team1, DataStats stats )
	{
		int x = 0;

		int scoreA = 0;
		int scoreB = 0;

		// Scan all stats
		foreach ( Stat stat in stats )
		{
			// Track score
			if ( stat.IsPoint )
			{
				x++;

				int score1 = stat.Point.Score1;
				int score2 = stat.Point.Score2;

				scoreA = team1 ? score1 : score2;
				scoreB = team1 ? score2 : score1;
			}
			// Timeout
			else if ( stat.IsUpdate && (stat.Selector == Stats.TimeoutKey) )
			{
				PlotTimeout( team1, stat, x, scoreA, scoreB );
			}
			// Substitution (primary only)
			else if ( team1 && stat.IsLineup && (stat.Selector == Stats.SubKey) )
			{
				PlotSub( true, stat, x, scoreA, scoreB );
			}
		}
	}

	// Adds Timeout annotation to line series
	private void PlotTimeout( bool team1, Stat stat, int x, int scoreA, int scoreB )
	{
		MomentumDot dot = CreateDot( team1, MomentumDot.DotType.Timeout, stat, x, scoreA, scoreB );

		View.AddAnnotation( dot );
	}

	// Adds Substitution annotation to line series
	private void PlotSub( bool team1, Stat stat, int x, int scoreA, int scoreB )
	{
		MomentumDot dot = CreateDot( team1, MomentumDot.DotType.Substitution, stat, x, scoreA, scoreB );

		View.AddAnnotation( dot );
	}

	// Creates annotation dot of specified type
	private static MomentumDot CreateDot( bool team1, MomentumDot.DotType type, Stat stat, int x, int scoreA, int scoreB )
	{
		// Momentum Y value
		int y = (scoreA - scoreB);

		// Populate data
		MomentumDot dot = new()
		{
			Type = type,
			Stat = stat,

			X = x,
			Y = y,

			ScoreA = scoreA,
			ScoreB = scoreB,

			Match = stat.Set.Match
		};

		dot.Init( team1 );

		return dot;
	}
}

//
