﻿/*
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC.
 */

using DXLib.UI.Layout;
using DXLib.UI.Container;
using DXLib.UI.Control.Button;

using DXLib.UI.Form;
using DXLib.UI.Form.Control;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Content area for popup menu used to adjust options for a heat map. Options include adjusting the number of map areas
 * on both the X and Y axis.
 */
public class HeatOptions : DXScroll
{
	/* Constants */
	public const int MinAreas = 3;
	public const int MaxAreas = 12;
	public const int StepAreas = 3;

	// Defaults
	public const int DefaultAreasX = 6;
	public const int DefaultAreasY = 6;

	/* Events */
	public Action Confirmed;

	/* Properties */
	public int AreasX => GetValue( "areasx" );
	public int AreasY => GetValue( "areasy" );

	/* Fields */
	private DXVerticalLayout layout;
	private DXFormControlList controls;

	/* Methods */

	// Post-construction initialization
	public void Init()
	{
		bool ios = DXDevice.IsIOS;

		Orientation = ScrollOrientation.Vertical;

		// Underlying layout
		layout = new DXVerticalLayout
		{
			Spacing = (ios ? 10 : 4)
		};

		// Control list
		controls = new DXFormControlList
		{
			Title = DXString.Get( "heat.areas" )
		};

		// Court length
		controls.Add( new DXStepper
		{
			Key = "areasx",
			Title = "heat.areasx",
			Number = DefaultAreasX,
			MinValue = MinAreas,
			MaxValue = MaxAreas,
			Step = StepAreas,
			IsLooping = true
		}, false, false );

		// Court width
		controls.Add( new DXStepper
		{
			Key = "areasy",
			Title = "heat.areasy",
			Number = DefaultAreasY,
			MinValue = MinAreas,
			MaxValue = MaxAreas,
			Step = StepAreas,
			IsLooping = true
		}, false, false );

		controls.Init();

		layout.Add( controls.View );

		// OK
		DXButton okBtn = new()
		{
			Resource = "alert.ok",
			Type = DXButton.ButtonType.Action,

			Margin = DXUtils.Top( -20 ),

			ButtonTapped = OnOkTapped
		};

		okBtn.Init();
		
		layout.Add( okBtn );

		WidthRequest = 250;
		HeightRequest = 250;

		Content = layout;
	}

	// Returns value for specified control
	private int GetValue( string key )
	{
		return Convert.ToInt32( controls.GetObject( key ) );
	}

	/* Event Callbacks */

	// User confirmed, callback listener
	private void OnOkTapped( object sender )
	{
		Confirmed?.Invoke();
	}
}

//
