﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI.Container;
using DXLib.Utils;

namespace iStatVball3;

/*
 * Report card for the heat map report type. Contains a report header and heat map. Adds a settings button to the header.
 */
public class HeatCard : ReportCard
{
	/* Properties */
	public HeatMap Map { get; private set; }
	public HeatView View { get; private set; }

	// Area square dimensions
	public int AreasX { get; private set; }
	public int AreasY { get; private set; }

	/* Fields */

	// Options popup
	private HeatOptions options;

	/* Methods */
	public HeatCard( JsonReport json, string dimension ) : base( json, dimension )
	{}

	// Post-construction initialization
	public override void Init()
	{
		// Extra button in header
		Options = "settings";

		base.Init();

		// Underlying component
		View = new HeatView
		{
			Margin = new Thickness( (Pad - 10), 10, (Pad - 5), (Pad + 10) ),

			LoadComplete = OnLoadComplete
		};

		layout.Add( View, 1, 1 );

		// Helper class
		Map = new HeatMap
		{
			Card = this,
			View = View
		};

		View.Init( json.Heat );

		// Setup chart
		Map.Init( json.Heat );

		AreasX = HeatOptions.DefaultAreasX;
		AreasY = HeatOptions.DefaultAreasY;

		// Set initial map dimensions
		View.SetAreas( AreasX, AreasY );
		Map.UpdateAreas( AreasX, AreasY );
	}

	// Updates all data displayed on this card
	public override async Task UpdateData( DataConfig config )
	{
		await base.UpdateData( config );

		// Update underlying control			
		Map.UpdateData( config, stats );
	}

	/* Event Callbacks */

	// User tapped extra header button
	protected override void OnOptionsTapped( View view )
	{
		// Lazily create popup
		if ( popup == null )
		{
			// Configure
			options = new HeatOptions
			{
				Confirmed = OnConfirmed
			};

			options.Init();

			// Create popup
			popup = new DXPopup( "report.options", options )
			{
				AutoSize = true,
				PopupClosed = OnClosed
			};
		}

		// Show
		popup.ShowFromView( view );
	}

	// User tapped OK
	private void OnConfirmed()
	{
		AreasX = options.AreasX;
		AreasY = options.AreasY;

		// Dynamically change map dimensions
		Map.UpdateAreas( AreasX, AreasY );
		View.SetAreas( AreasX, AreasY );

		// Calculate and draw
		Map.Update();

		popup.Hide();
	}

	// Reset sticky state
	private void OnClosed()
	{
		header.Reset();
	}

	/* Layout */

	// Device/orientation specific layout
	public override void UpdateLayout( LayoutType type )
	{
		base.UpdateLayout( type );

		// WORKAROUND: MAUI issues
		switch ( type )
		{
			case LayoutType.MobilePortrait:
			{
				View.Margin = new Thickness( Pad, 10, Pad, Pad );
				break;
			}
			case LayoutType.MobileLandscape:
			{
				double safe = DXDevice.SafeArea().Left;

				View.Margin = new Thickness( (safe + Pad), 10, (safe + Pad), Pad );
				break;
			}
		}

		// Update children
		View.UpdateLayout( type );
	}
}

//
