﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;
using DXLib.UI.Control;
using DXLib.UI.Container;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Segment bar interface for choosing the report type in the Report Builder.
 */
public class BuilderType : DXContent
{
	/* Constants */
	private static readonly double DefaultSegmentWd = DXDevice.IsTablet ? 85 : 79;
	private const double DefaultSegmentHt = 65;
	
	/* Events */
	public Action<string> TypeChanged { get; set; }

	/* Properties */
	public string SelectedType { get => segmentBar.SelectedKey; set => segmentBar.SelectedKey = value; }
	
	/* Fields */
	private readonly DXSegmentBar segmentBar;

	// Config
	private JsonDashboard jsonDashboard;

	/* Methods */
	public BuilderType()
	{
		BackgroundColor = DXColors.Dark3;
		
		Horizontal = LayoutOptions.Fill;
		Vertical = LayoutOptions.Fill;
		
		// Grid/Bar/Shot/Heat
		segmentBar = new DXSegmentBar
		{
			Mode = DXSegmentBar.SegmentMode.IconText,

			FillColor = DXColors.Dark3,
			TextColor = DXColors.Light4,
			IconColor = DXColors.Light4,
			SelectColor = DXColors.Light4,

			SegmentWd = DefaultSegmentWd,
			SegmentHt = DefaultSegmentHt,
			SegmentFontSize = 12,

			Margin = 0,
			
			HeightRequest = DefaultSegmentHt,
			WidthRequest = ((DefaultSegmentWd + (DXSegmentBar.SelectionWd * 2)) * 5),
			
			Horizontal = LayoutOptions.Center,
			Vertical = LayoutOptions.Center,

			Selected = OnSelected
		};
	}

	// Post-construction initialization
	public void Init( JsonDashboard json )
	{
		jsonDashboard = json;

		segmentBar.Init();

		// Populate selection interface
		AddSegment( AnalyzeKeys.GridKey );
		AddSegment( AnalyzeKeys.BarKey );
		AddSegment( AnalyzeKeys.ShotKey );
		AddSegment( AnalyzeKeys.HeatKey );
		AddSegment( AnalyzeKeys.TrendKey );

		segmentBar.SelectKey( AnalyzeKeys.GridKey );
		segmentBar.DisableAll();

		// Must explicitly enable applicable types
		foreach ( string key in json.Types )
		{
			// Momentum reuses Trend segment
			if ( key != AnalyzeKeys.MomentumKey )
			{
				segmentBar.Disable( key, false );
			}
		}

		Content = segmentBar;
	}

	// Adds segment to segment bar for specified key
	private void AddSegment( string key, double adjust = 0 )
	{
		string icon = $"{key}.png";// GetIcon( key );
		string resource = $"builder.{key}";

		segmentBar.AddSegmentIconText( key, icon, resource );
	}

	// Enables/disables report types based on query
	public void UpdateData( DataConfig config, DataStats stats )
	{
		// Shot Chart, Heat Map only applicable for RallyFlow
		if ( !stats.HasRally )
		{
			segmentBar.Disable( AnalyzeKeys.ShotKey, true );
			segmentBar.Disable( AnalyzeKeys.HeatKey, true );
		}

		bool hasMomentum = jsonDashboard.HasReport( AnalyzeKeys.MomentumKey );

		// Trend Graph and Momentum Chart share same segment
		bool hasTrend = jsonDashboard.HasReport( AnalyzeKeys.TrendKey ) || hasMomentum;

		// Trend NA for org scope, Momentum only applicable for set scope scoring
		bool trendInvalid = (config.Scope == DataConfig.OrganizationScope) || ((config.Scope == DataConfig.SetScope) && !hasMomentum);

		segmentBar.Disable( AnalyzeKeys.TrendKey, (!hasTrend || trendInvalid) );
	}

	/* Event Callbacks */

	// User selected new type
	private void OnSelected( string key )
	{
		TypeChanged?.Invoke( key );
	}

	/* Layout */

	// Device/orientation specific layout
	public override void UpdateLayout( LayoutType type )
	{
		Thickness safeArea = DXDevice.SafeArea();
		double leftPad = DXDevice.IsMobileLandscape() ? safeArea.Left : 10;

		// Must account for left notch
		Padding = new Thickness( leftPad, 0, 0, 0 );
		
		// Dynamic size
		segmentBar.WidthRequest = ((DefaultSegmentWd + (DXSegmentBar.SelectionWd * 2)) * 5);
	}
}

//
