﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;
using DXLib.UI.Layout;
using DXLib.UI.Control;

using DXLib.Utils;

namespace iStatVball3;

/*
 * Segment bar interface for selecting a team in the ReportBuilder.
 */
public class BuilderTeam : DXGridLayout
{
	/* Constants */
	public static readonly Thickness BasePadding = new( 15, 5, 15, 5 );
	
	// Segment keys
	public const string Team1Key = "team1";
	public const string Team2Key = "team2";

	/* Events */
	public Action<bool> TeamChanged { get; set; }

	/* Properties */
	public bool IsTeam1
	{
		get => segmentBar.SelectedKey == Team1Key;
		set => segmentBar.SelectedKey = value ? Team1Key : Team2Key;
	}

	/* Fields */
	private readonly DXSegmentBar segmentBar;

	// Query configuration
	private ReportConfig reportConfig;

	/* Methods */
	public BuilderTeam()
	{
		BackgroundColor = DXColors.Dark3;

		Padding = 0;				// Dynamic
		RowSpacing = 0;
		ColumnSpacing = 0;

		Horizontal = LayoutOptions.Fill;
		Vertical = LayoutOptions.Fill;

		// REQUIRED
		IgnoreSafeArea = true;
		
		// Team 1/2
		segmentBar = new DXSegmentBar
		{
			Mode = DXSegmentBar.SegmentMode.Text,

			FillColor = DXColors.Dark3,
			TextColor = DXColors.Light4,
			SelectColor = DXColors.Light4,

			SegmentHt = 40,						// Width set dynamically
			SegmentFontSize = 15,

			Margin = 0,
			
			Horizontal = LayoutOptions.Center,
			Vertical = LayoutOptions.Center,

			Selected = OnSelected
		};

		segmentBar.Init();

		// Text set dynamically later
		segmentBar.AddSegmentText( Team1Key );
		segmentBar.AddSegmentText( Team2Key );
		
		// Always start with Team 1
		segmentBar.SelectKey( Team1Key );
	}

	// Post-construction initialization
	public void Init( JsonDashboard json, ReportConfig config )
	{
		reportConfig = config;

		// Opponent may not be applicable
		if ( json.Teams < 2 )
		{
			segmentBar.Disable( "team2", true );
		}

		// May be entering from opponent dashboard
		IsTeam1 = config.IsTeam1;
	}

	// Opponent not available for legacy data sets
	public void UpdateData( DataStats stats )
	{
		if ( !stats.HasRally )
		{
			segmentBar.Disable( "team2", true );
		}
	}

	/* Event Callbacks */

	// User changed team selection
	private void OnSelected( string key )
	{
		bool isTeam1 = (key == "team1");

		TeamChanged?.Invoke( isTeam1 );
	}

	// Generates display name for Team 2
	private string GetTeam2Name( bool abbrev )
	{
		string scope = reportConfig.Scope;

		// Single opponent
		if ( scope is DataConfig.SetScope or DataConfig.MatchScope )
		{
			Opponent opponent = reportConfig.Team2;

			// Anonymous
			if ( opponent == null )
			{
				return DXString.Get( abbrev ? "analyze.abbrev.opp" : "analyze.opp" );
			}
			
			// Named
			return abbrev ? opponent.Abbreviation : opponent.ShortName;
		}

		// Multiple opponents
		return DXString.Get( abbrev ? "analyze.abbrev.opps" : "analyze.opps" );
	}

	/* Layout */

	// Device/orientation specific layout
	public override void UpdateLayout( LayoutType type )
	{
		Clear();

		double rightPad = DXDevice.IsMobileLandscape() ? DXDevice.SafeArea().Right : 0;
		Thickness extraPad = new Thickness( 0, 0, rightPad, 0 );

		// Account for right-hand notch
		Padding = DXUtils.AddPadding( BasePadding, extraPad );
		
		string text1;
		string text2;

		double segmentWd;

		// Size and labels change based on orientation
		if ( type == LayoutType.Landscape )
		{
			text1 = reportConfig.Organization.Name;
			text2 = GetTeam2Name( false );

			segmentWd = 175;
		}
		else
		{
			text1 = reportConfig.Organization.Abbreviation;
			text2 = GetTeam2Name( true );

			segmentWd = DXDevice.IsMobile ? 130 : 150;
		}

		// Update labels
		segmentBar.SetText( "team1", text1.ToUpper() );
		segmentBar.SetText( "team2", text2.ToUpper() );
		
		// Update size
		segmentBar.SegmentWd = segmentWd;
		segmentBar.WidthRequest = ((segmentWd + (DXSegmentBar.SelectionWd * 2)) * 2);

		Add( segmentBar, 0, 0 );
	}
}

//
