﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI;
using DXLib.UI.Layout;
using DXLib.UI.Container;

using DXLib.UI.Control.Button;
using DXLib.UI.Control.Checkbox;

namespace iStatVball3;

/*
 * View for selecting filter options for a query builder dimension. The list of options is dependent on the currently
 * selected dimension (player, rotation, etc). A root "TEAM" option is provided that (de)selects all child items. The
 * list of child items contains one entry for each dimension object present in the data set.
 */
public class BuilderFilter : DXGridLayout
{
	/* Constants */
	public const double DefaultWd = 250;
	public const double DefaultHt = 370;
	
	/* Events */
	public Action Selected { get; set; }

	/* Properties */

	// Allow team-only selection?
	public bool AllowEmpty { get => treeView.AllowEmpty; set => treeView.AllowEmpty = value; }

	/* Fields */
	private readonly DXCheckboxNode treeView;
	private readonly DXButton okBtn;

	// Config
	private readonly DataDimension.Level level;
	private string dimension;

	// Filter data
	private List<object> filterList;
	private List<string> selectedKeys;

	/* Methods */
	public BuilderFilter( DataDimension.Level level )
	{
		this.level = level;

		BackgroundColor = DXColors.Light2;

		Padding = 0;
		RowSpacing = 0;

		Horizontal = LayoutOptions.Fill;
		Vertical = LayoutOptions.Fill;
		
		// Fixed size
		WidthRequest = DefaultWd;
		HeightRequest = DefaultHt;

		// 2 rows
		AddStarRow();			// Tree view
		AddFixedRow( 56 );		// OK button

		// Scrollable list
		DXScroll scroll = new()
		{
			BackgroundColor = DXColors.Light4,
			Padding = new Thickness( 10, 10, 10, 10 ),
			Orientation = ScrollOrientation.Vertical
		};

		// Root/child nodes
		treeView = new DXCheckboxNode
		{
			SelectionChanged = OnFilterChanged
		};

		scroll.Content = treeView;

		Add( scroll, 0, 0 );

		// OK
		okBtn = new DXButton
		{
			Resource = "alert.ok",
			Type = DXButton.ButtonType.Action,

			Horizontal = LayoutOptions.Center,
			Vertical = LayoutOptions.Center,

			ButtonTapped = OnOkTapped
		};

		okBtn.Init();
		
		Add( okBtn, 0, 1 );
	}

	// Post-construction initialization
	public void Init()
	{
		// Populate tree view
		LoadData();

		// Default to ALL selected
		treeView.SelectAll( true );
	}

	// Determines if filter keys are selected (all selected is no filter)
	public bool HasFilter()
	{
		return (selectedKeys.Count != filterList.Count);
	}

	// Returns list of objects to be filtered against
	public List<object> GetFilter()
	{
		return KeyDimension.GetFilter( dimension, treeView.GetSelectedObjects() );
	}

	// Returns comma-delimited list of selected filter values
	public string BuildList()
	{
		List<object> filter = GetFilter();

		return KeyDimension.GetValues( dimension, filter );
	}

	/* Load */

	// Populate tree view
	private void LoadData()
	{
		KeyDimension.Load( dimension, filterList, treeView, AllowEmpty );
	}

	/* Update */

	// (Re)builds filter based on new data
	public void UpdateData( DataConfig config, DataStats stats )
	{
		dimension = (level == DataDimension.Level.Outer) ? config.OuterDim : config.InnerDim;

        // Build filter
        filterList = KeyDimension.Update( dimension, config, stats );

        // Reset
        Init();
    }

	/* Event Callbacks */

	// User changed filter selection
	private void OnFilterChanged()
	{
		selectedKeys = treeView.GetSelectedKeys();

		// OK requires selection
		if ( okBtn != null )
		{
			okBtn.IsDisabled = (selectedKeys.Count == 0) && !AllowEmpty;
		}
	}

	// User confirmed filter selection
	private void OnOkTapped( object sender )
	{
		Selected?.Invoke();
	}
}

//
