﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.Email;
using DXLib.Utils;

namespace iStatVball3;

/*
 * Exports a stats analysis report to PDF format and sends it as an email attachment.
 */
public class PDFExporter : Exporter
{
    /* Constants */
    private const string FilePrefix = "istatvball3";
    private const string FileExt = "pdf";
    private const string FileType = "application/pdf";

    /* Methods */

    // Creates PDF document then sends via email
    public async Task<DXEmail.Result> Export( ExportConfig config )
    {
        PDFReport report = new();

        // Create PDF document data
        byte[] pdf = await report.Create( config );

        bool local = config.IsLocal;

        // Configure email
        DXEmailConfig emailConfig = new()
        {
            // Local/remote
            IsLocal = local,

            // From:
            FromAddress = config.FromAddress,
            FromName = config.FromName,

            // To:
            ToAddresses = config.ToAddresses,
            ToNames = config.ToNames,

            // Local use only
            Subject = config.Subject,
            Body = DXString.Get( "export.body", config.Title ),

            // Template
            TemplateId = "d-0920a5e640724679b1cd3aab86f278b5",
            TemplateData = config,

            // Attachment
            FileRaw = pdf,
            FileName = CreateFilename( config ),
            FileData = DXUtils.ToBase64( pdf ),
            FileType = FileType
        };

        // Async send
        DXEmail.Result result = await DXEmail.Send( emailConfig );
        
        return result;
    }

    // Generates canonical filename for export file
    private static string CreateFilename( ExportConfig config )
    {
        string dateStr = DXUtils.FilenameFromDate( config.Date );

        // 'istatvball3_boxscore_241027.pdf'
        string fullName = $"{FilePrefix}_{config.Title}_{config.Team}_{dateStr}.{FileExt}";
        string filename = DXUtils.ToFilename( fullName );

        return filename;
    }
}

//
