﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.Utils;

namespace iStatVball3;
	
/*
 * Accumulates all scoring related metrics for the Rotation dimension.
 */
public static class MetricsScoringRotation
{
	// Accumulates all dimensional variables from specified stats list
	public static void AccumulateDimension( DataConfig config, DataStats stats )
	{
		// Create row for each rotation
		for ( byte rotation = 1; rotation <= Lineup.BaseEntries; rotation++ )
		{
			BuildDimension( config, rotation );
		}

		// Stats list already filtered for scoring
		foreach ( Stat stat in stats )
		{
			// Earned points/unforced errors come from action (+/- NA)
			if ( stat.IsAction )
			{
				AccumulatePointsErrors( config, stat );
			}
		}
	}

	// Accumulates all earned point and unforced error metrics
	private static void AccumulatePointsErrors( DataConfig config, Stat stat )
	{
		bool earned = stat.IsEarned();
		bool error = stat.IsError;

		bool earnable = stat.IsEarnable();
		bool errorable = stat.IsErrorable();

		if ( config.GetMetrics( stat.Rotation ) is MetricsScoring metrics )
		{
			// Earned point attempt
			if ( earnable )
			{
				metrics.PointAttempts += 1;
			}

			// Unforced error attempt
			if ( errorable )
			{
				metrics.ErrorAttempts += 1;
			}

			// Earned point
			if ( earned )
			{
				metrics.Points += 1;
			}
			// Unforced error
			else if ( error )
			{
				metrics.Errors += 1;
			}
		}

		if ( config.Totals is MetricsScoring totals )
		{
			// Team totals
			if ( earnable )
			{
				totals.PointAttempts += 1;
			}

			if ( errorable )
			{
				totals.ErrorAttempts += 1;
			}

			if ( earned )
			{
				totals.Points += 1;
				totals.PointsTeam += 1;
			}
			else if ( error )
			{
				totals.Errors += 1;
				totals.ErrorsTeam += 1;
			}
		}

		// Track team totals used for each line item
		for ( byte rotation = 1; rotation <= Lineup.BaseEntries; rotation++ )
		{
			if ( config.GetMetrics( rotation ) is MetricsScoring totalMetrics )
			{
				if ( earned )
				{
					totalMetrics.PointsTeam += 1;
				}
				else if ( error )
				{
					totalMetrics.ErrorsTeam += 1;
				}
			}
		}
	}

	// Builds dimension (row) entry for specified rotation
	private static void BuildDimension( DataConfig config, byte rotation )
	{
		// Check for dimension filter (access control NA)
		if ( (config.OuterFilter == null) || config.OuterFilter.Contains( rotation ) )
		{
			// Keyed by rotation number
			string key = rotation.ToString();
			string label = $"{DXString.Get( "dimension.rotation" )} {rotation}";

			var metricsList = config.Dimension.Metrics;

			// Prevent duplicates
			if ( !metricsList.ContainsKey( key ) )
			{
				// Create new dimension
				MetricsScoring metrics = new()
				{
					Type = KeyDimension.RotationKey,
					TypeKey = null,
					TypeLabel = null,

					TypeObject = rotation,

					Key = key,
					Label = label
				};

				metricsList.Add( key, metrics );
			}
		}
	}
}

//
