﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

namespace iStatVball3
{
	/*
	 * Encapsulates all variables and statistics for Scoring related metrics including
	 * earned points, unforced errors, point differential, and +/- scoring.
	 */
	public class MetricsScoring : DataMetrics
	{
		/* Variables */

		// Earned points
		public float Points { get; set; }
		public float PointsTeam { get; set; }
		public float PointAttempts { get; set; }

		// Unforced errors
		public float Errors { get; set; }
		public float ErrorsTeam { get; set; }
		public float ErrorAttempts { get; set; }

		// +/-
		public float Plus { get; set; }
		public float Minus { get; set; }
		public float Played { get; set; }

		/* Statistics */

		// Earned points
		public float PointsPerSet { get; set; }
		public float PointPercent { get; set; }
		public float PointTeamPercent { get; set; }

		// Unforced errors
		public float ErrorsPerSet { get; set; }
		public float ErrorPercent { get; set; }
		public float ErrorTeamPercent { get; set; }

		// Point differential
		public float Differential { get; set; }
		public float DifferentialPerSet { get; set; }

		// +/-
		public float PlusMinus { get; set; }
		public float PlusMinusNorm { get; set; }

		public float PlayedPercent { get; set; }

		/* Methods */
		public MetricsScoring( bool total = false ) : base( total )
		{}

		/* Report Accumulation */

		// Accumulates all dimensional variables from specified stats list
		public static void AccumulateDimension( DataConfig config, DataStats stats )
		{
			config.Dimension = new DataDimension();
			config.Totals = new MetricsScoring( true );

			// Player dimension
			switch ( config.OuterDim )
			{
				case KeyDimension.PlayerKey:
				{
					MetricsScoringPlayer.AccumulateDimension( config, stats );
					break;
				}
				case KeyDimension.RotationKey:
				{
					MetricsScoringRotation.AccumulateDimension( config, stats );
					break;
				}
			}
		}

		/* Dashboard Accumulation */

		// Accumulates minimal set of metrics needed for dashboard
		public override void Accumulate( DataConfig config, Stat stat, int playerNum = 1 )
		{
			// Earned point
			if ( stat.IsEarned() )
			{
				Points += 1;
			}
			// Unforced error
			else if ( stat.IsError )
			{
				Errors += 1;
			}
		}

		/* Calculate */

		// Calculates all statistics for pre-accumulated variables
		public override void Calculate()
		{
			float sets = SetsPlayed;
			bool hasSets = (sets > 0);

			// Earned points
			Points = hasSets ? Points : FloatNA;
			PointAttempts = hasSets ? PointAttempts : FloatNA;

			PointsPerSet = hasSets ? (Points / sets) : FloatNA;
			PointPercent = (PointAttempts > 0) ? (Points / PointAttempts) : FloatNA;
			PointTeamPercent = (PointsTeam > 0) ? (Points / PointsTeam) : FloatNA;

			// Unforced errors
			Errors = hasSets ? Errors : FloatNA;
			ErrorAttempts = hasSets ? ErrorAttempts : FloatNA;

			ErrorsPerSet = hasSets ? (Errors / sets) : FloatNA;
			ErrorPercent = (ErrorAttempts > 0) ? (Errors / ErrorAttempts) : FloatNA;
			ErrorTeamPercent = (ErrorsTeam > 0) ? (Errors / ErrorsTeam) : FloatNA;

			// Point differential
			Differential = hasSets ? (Points - Errors) : FloatNA;
			DifferentialPerSet = hasSets ? (PointsPerSet - ErrorsPerSet) : FloatNA;

			// +/-
			Plus = hasSets ? Plus : FloatNA;
			Minus = hasSets ? Minus : FloatNA;
			Played = hasSets ? Played : FloatNA;

			PlusMinus = hasSets ? (Plus - Minus) : FloatNA;
			PlayedPercent = (Played / TotalPoints);
			PlusMinusNorm = (PlusMinus / PlayedPercent);
		}
	}
}

//
