﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.Data;
using DXLib.Utils;

namespace iStatVball3;

/*
 * Encapsulates all variables for displaying a match results table. Each data row represents a match with match info,
 * result, and set scores.
 */
public class MetricsResults : DataMetrics
{
    /* Properties */

    // Fields
    public string Result_Opponent { get; set; }
    public string Result_Date { get; set; }
    public string Result_Result { get; set; }
    public string Result_Sets { get; set; }

    // Sets 1-5
    public string Result_Set1 { get; set; }
    public string Result_Set2 { get; set; }
    public string Result_Set3 { get; set; }
    public string Result_Set4 { get; set; }
    public string Result_Set5 { get; set; }

    /* Methods */
    public MetricsResults()
    {}

    // Populates results object from specified match
    public MetricsResults( Match match )
    {
        Result_Opponent = match.GetOpponentName();
        Result_Date = DXUtils.LabelFromDate( match.MatchTime );
        Result_Result = DXString.Get( match.GetResultRsrc() );
        Result_Sets = match.GetSetResult();

        IList<Set> sets = match.Sets;
        int count = sets.Count;

        // 1-5 sets
        Result_Set1 = (count > 0) ? sets[0].GetScore() : DXData.NaN;
        Result_Set2 = (count > 1) ? sets[1].GetScore() : DXData.NaN;
        Result_Set3 = (count > 2) ? sets[2].GetScore() : DXData.NaN;
        Result_Set4 = (count > 3) ? sets[3].GetScore() : DXData.NaN;
        Result_Set5 = (count > 4) ? sets[4].GetScore() : DXData.NaN;
    }

    // Loads all match results for specified scope
    public static MetricsResults LoadData( DataConfig config )
    {
        // Aggregate matches
        List<Match> matches = GetMatches( config );

        // Sort
        List<Match> sortedMatches = matches.OrderByDescending( m => m.MatchTime ).ToList();

        List<MetricsResults> results = new( sortedMatches.Count );
        bool scrimmages = Shell.Settings.AnalyzeScrimmage;

        // Add all ended, non-scrimmage matches in scope
        foreach ( Match match in sortedMatches )
        {
            if ( match.IsEnded )
            {
                if ( scrimmages || !match.IsScrimmage )
                {
                    MetricsResults result = new( match );
                    results.Add( result );
                }
            }
        }

        // Create metrics
        MetricsResults data = new()
        {
            Objects = results.Cast<object>().ToList()
        };

        return data;
    }

    // Unused
    public override void Accumulate( DataConfig config, Stat stat, int playerNum = 1 )
    {
        throw new NotImplementedException();
    }

    // Unused
    public override void Calculate()
    {
        throw new NotImplementedException();
    }
}

//
