﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

namespace iStatVball3;

/*
 * Encapsulates all variables and statistics for Match win/loss related metrics. Unlike most metrics, match metrics are
 * accumulated at the match/set level NOT from raw stats. Also provides utility methods for accumulating and calculating
 * the match stats.
 */
public class MetricsMatch : DataMetrics
{
    /* Properties */

    // Variables
    public int Matches_Won { get; set; }
    public int Matches_Lost { get; set; }
    public int Matches_Tied { get; set; }

    public int Sets_Won { get; set; }
    public int Sets_Lost { get; set; }
    public int Sets_Tied { get; set; }

    public int Points_Won { get; set; }
    public int Points_Lost { get; set; }
    public int Points_Tied { get; set; }

    // Calculated Variables
    public int Matches_Total { get; set; }
    public int Sets_Total { get; set; }
    public int Points_Total { get; set; }

    // Statistics
    public float Matches_WinPercent { get; set; }
    public float Sets_WinPercent { get; set; }
    public float Points_WinPercent { get; set; }

    /* Fields */
    private bool accumulated;

    /* Methods */
    public MetricsMatch( bool total = false ) : base( total )
    {}

    // Loads match win/loss data for use in static tables
    public static MetricsMatch LoadData( DataConfig config )
    {
        MetricsMatch metrics = new();

        // Load and accumulate matches
        metrics.Accumulate( config );

        return metrics;
    }

    // Accumulates all match win/loss related variables
    public override void Accumulate( DataConfig config, Stat stat, int playerNum = 1 )
    {
        // Only done once
        if ( !accumulated )
        {
            Accumulate( config );

            accumulated = true;
        }
    }

    // Used internally to accumulate all match win/loss related variables
    private void Accumulate( DataConfig config )
    {
        List<Match> matches = GetMatches( config );

        // Maybe an unsupported scope
        if ( matches != null )
        {
            // Always NA
            Points_Tied = IntNA;

            bool scrimmages = Shell.Settings.AnalyzeScrimmage;

            // Accumulate each match
            foreach ( Match match in matches )
            {
                if ( scrimmages || !match.IsScrimmage )
                {
                    Match.MatchResult matchResult = (Match.MatchResult)match.Result;

                    // Matches won/lost
                    switch ( matchResult )
                    {
                        case Match.MatchResult.Won: Matches_Won++; break;
                        case Match.MatchResult.Lost: Matches_Lost++; break;
                        case Match.MatchResult.Tied: Matches_Tied++; break;
                        default: break;
                    }

                    // Sets won/lost
                    foreach ( Set set in match.Sets )
                    {
                        Set.SetResult setResult = (Set.SetResult)set.Result;

                        switch ( setResult )
                        {
                            case Set.SetResult.Won: Sets_Won++; break;
                            case Set.SetResult.Lost: Sets_Lost++; break;
                            case Set.SetResult.Tied: Sets_Tied++; break;
                            default: break;
                        }

                        // Points won/lost
                        Points_Won += set.Points1;
                        Points_Lost += set.Points2;
                    }
                }
            }
        }
    }

    // Calculates all match win/loss statistics from accumulated variables
    public override void Calculate()
    {
        // Calculated variables
        Matches_Total = (Matches_Won + Matches_Lost + Matches_Tied);
        Sets_Total = (Sets_Won + Sets_Lost + Sets_Tied);
        Points_Total = (Points_Won + Points_Lost + Points_Tied);

        // Statistics
        Matches_WinPercent = ((float)Matches_Won / (float)Matches_Total);
        Sets_WinPercent = ((float)Sets_Won / (float)Sets_Total);
        Points_WinPercent = ((float)Points_Won / (float)Points_Total);
    }
}

//
