﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI.Control.Checkbox;

namespace iStatVball3;

/*
 * Provides utility methods for working with an analyze By Setter key. Only applicable for hitting stats. Note that the
 * player is NOT the actual key.
 */
public class KeySetter
{
	/* Properties */
	public string UniqueId { get; set; }
	public string ObjectName { get; set; }

	/* Methods */
	public KeySetter( Stat stat )
	{
		// Player ID
		UniqueId = GetKey( stat );

		// 'John Doe'
		ObjectName = GetLabel( stat );
	}

	// Tests for equality between two key objects
	public override bool Equals( object obj )
	{
		return (obj is KeySetter setter) && setter.UniqueId.Equals( UniqueId );
	}

	// Generates unique hash code for lists
	public override int GetHashCode()
	{
		return UniqueId.GetHashCode();
	}

	// Returns unique key for dimension analysis
	public static string GetKey( Stat stat )
	{
		return stat.Player2Id;
	}

	// Creates display label for setter from specified stat
	public static string GetLabel( Stat stat )
	{
		return stat.Player2?.FullName;
	}

	// Returns list of filter objects for this dimension
	public static List<object> GetFilter( List<object> objects )
	{
		return objects;
	}

	// Determines if setter from specified stat passes given filter
	public static bool Filter( List<object> filter, Stat stat )
	{
		return (filter == null) || ((stat.Player2 != null) && filter.Contains( new KeySetter( stat ) ));
	}

	// Returns comma-delimited list of selected filter values
	public static string GetValues( List<object> filter )
	{
		int count = filter.Count;
		string[] values = new string[count];

		// Build comma delimited list
		for ( int i = 0; i < count; i++ )
		{
			object obj = filter[i];

			values[i] = (obj as KeySetter)?.ObjectName;
		}

		return string.Join( ", ", values );
	}

	// Loads filter tree for dimension
	public static void Load( DXCheckboxNode tree, List<object> filter )
	{
		tree.SetChildObjects( filter );
	}

	// Updates filter list for dimension
	public static List<object> Update( DataStats stats )
	{
		List<KeySetter> list = [];

		// Build list of each setter present in data
		foreach ( Stat stat in stats )
		{
			if ( stat.Player2 != null )
			{
				KeySetter setter = new( stat );

				if ( !list.Contains( setter ) )
				{
					list.Add( setter );
				}
			}
		}

		// Alpha sort
		return list.OrderBy( ks => ks.ObjectName ).Cast<object>().ToList();
	}
}

//
