﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI.Control.Checkbox;
using DXLib.Utils;

namespace iStatVball3;

/*
 * Provides utility methods for working with a key for analysis by court row (front/back row).
 */
public class KeyRow
{
	/* Properties */
	public string UniqueId { get; set; }
	public string ObjectName { get; set; }

	/* Methods */
	public KeyRow( Stat stat )
	{
		UniqueId = GetKey( stat );
		ObjectName = GetLabel( stat );
	}

	// Tests for equality between two row objects
	public override bool Equals( object obj )
	{
		return (obj is KeyRow row) && row.UniqueId.Equals( UniqueId );
	}

	// Generates unique hash code for lists
	public override int GetHashCode()
	{
		return UniqueId.GetHashCode();
	}

	// Returns unique key for stat analysis
	public static string GetKey( Stat stat )
	{
		return stat.CourtRow.ToString();
	}

	// Creates display label for specified row ('Front Row')
	public static string GetLabel( Stat stat )
	{
		string key = GetKey( stat );

		return DXString.GetLookupValue( "court.row", key );
	}

	// Returns list of filter objects for this dimension
	public static List<object> GetFilter( List<object> objects )
	{
		return objects;
	}

	// Determines whether specified row should be included in filter
	public static bool Filter( List<object> filter, Stat stat )
	{
		if ( Include( stat ) )
		{
			KeyRow row = new( stat );

			return (filter == null) || filter.Contains( row );
		}

		return false;
	}

	// Determines if specified stat relevant for court row dimension
	private static bool Include( Stat stat )
	{
		// Third ball attack only
		return ((stat.Action == Stats.ThirdKey) && (stat.Selector == Stats.AttackKey));
	}

	// Returns comma-delimited list of selected filter values
	public static string GetValues( List<object> filter )
	{
		int count = filter.Count;
		string[] values = new string[count];

		// Build comma delimited list
		for ( int i = 0; i < count; i++ )
		{
			KeyRow row = filter[i] as KeyRow;

			values[i] = $"{row?.ObjectName}";
		}

		return string.Join( ", ", values );
	}

	// Loads filter tree for dimension
	public static void Load( DXCheckboxNode tree, List<object> filter )
	{
		tree.SetChildObjects( filter );
	}

	// Updates filter list for dimension
	public static List<object> Update( DataStats stats )
	{
		List<KeyRow> list = new();

		// Build list of each relevant row present in data
		foreach ( Stat stat in stats )
		{
			if ( Include( stat ) )
			{
				KeyRow row = new( stat );

				if ( !list.Contains( row ) )
				{
					list.Add( row );
				}
			}
		}

		// Sort
		return list.OrderBy( kr => kr.ObjectName ).Cast<object>().ToList();
	}
}

//
