﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI.Control.Checkbox;
using DXLib.Utils;

namespace iStatVball3;

/*
 * Provides utility methods for working with an analyze By Rotation key. Note that the KeyRotation object is the actual
 * key.
 */
public class KeyRotation
{
	/* Constants */
	private static readonly string RotationStr = DXString.Get( "dimension.rotation" );

	/* Properties */
	public string UniqueId { get; set; }
	public string ObjectName { get; set; }

	/* Methods */
	public KeyRotation( int rotation )
	{
		UniqueId = rotation.ToString();
		ObjectName = $"{RotationStr} {rotation}";
	}

	// Tests for equality between two rotation objects
	public override bool Equals( object obj )
	{
		return (obj is KeyRotation rotation) && rotation.UniqueId.Equals( UniqueId );
	}

	// Generates unique hash code for lists
	public override int GetHashCode()
	{
		return UniqueId.GetHashCode();
	}

	// Returns unique key for stat analysis
	public static string GetKey( Stat stat, bool sideout )
	{
		int rotation = stat.GetRotation( sideout );

		return rotation.ToString();
	}

	// Creates display label for specified rotation ('Rotation N')
	public static string GetLabel( int rotation )
	{
		return $"{RotationStr} {rotation}";
	}

	// Creates display label from specified stat ('Rotation N')
	public static string GetLabel( Stat stat, bool sideout )
	{
		int rotation = stat.GetRotation( sideout );

		return GetLabel( rotation );
	}

	// Returns list of filter objects for this dimension
	public static List<object> GetFilter( List<object> objects )
    {
		List<int> filter = new( objects.Count );

		// Convert each Rotation object to int
		foreach ( object obj in objects )
		{
			KeyRotation rotationObj = obj as KeyRotation;
			int rotation = (int) DXUtils.ConvertToInt( rotationObj?.UniqueId )!;

			filter.Add( rotation );
		}

		return filter.Cast<object>().ToList();

	}

	// Determines whether specified rotation should be included in filter
	public static bool Filter( List<object> filter, Stat stat )
	{
		int rotation = stat.Rotation;

		return (rotation > 0) && ((filter == null) || filter.Contains( rotation ));
	}

	// Returns comma-delimited list of selected filter values
	public static string GetValues( List<object> filter )
	{
		int count = filter.Count;
		string[] values = new string[ count ];

		// Build comma delimited list
		for ( int i = 0; i < count; i++ )
		{
			object obj = filter[i];

			values[i] = $"{RotationStr} {obj}";
		}

		return string.Join( ", ", values );
	}

	// Loads filter tree for dimension
	public static void Load( DXCheckboxNode tree, List<object> filter )
	{
        List<object> list = new( filter.Count );

        // Convert ints to Rotations
        foreach ( object filterObj in filter )
        {
            list.Add( new KeyRotation( (int)filterObj ) );
        }

        tree.SetChildObjects( list );
    }

	// Updates filter list for dimension
	public static List<object> Update( DataStats stats )
	{
		List<int> list = [];

		// Build list of each rotation present in data
		foreach ( Stat stat in stats )
		{
			int rotation = stat.Rotation;

			if ( (rotation > 0) && !list.Contains( rotation ) )
			{
				list.Add( rotation );
			}
		}

		// Sort
		list.Sort();

		return list.Cast<object>().ToList();
	}
}

//
