﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI.Control.Checkbox;
using DXLib.Utils;

namespace iStatVball3;

/*
 * Provides utility methods for working with a key for analysis by serve receive rating. Note that the receive rating is
 * stored in the Value field of the third-attack Stat object.
 */
public class KeyRating
{
	/* Constants */
	private static readonly string RatingStr = DXString.Get( "shot.rating" );

	/* Properties */
	public string UniqueId { get; set; }
	public string ObjectName { get; set; }

	/* Methods */
	public KeyRating( Stat stat )
	{
		UniqueId = GetKey( stat );
		ObjectName = GetLabel( stat );
	}

	// Tests for equality between two rating objects
	public override bool Equals( object obj )
	{
		return (obj is KeyRating rating) && rating.UniqueId.Equals( UniqueId );
	}

	// Generates unique hash code for lists
	public override int GetHashCode()
	{
		return UniqueId.GetHashCode();
	}

	// Returns unique key for stat analysis
	public static string GetKey( Stat stat )
	{
		return GetValue( stat );
	}

	// Creates display label for specified rating ('Rating 0')
	public static string GetLabel( Stat stat )
	{
		string value = GetValue( stat );

		return $"{RatingStr} {value}";
	}

	// Returns byte rating value from specified stat as string
	static string GetValue( Stat stat )
	{
		int? rating = stat.Value;

		// Must map between 0-4 and 0-3
		if ( rating.HasValue && Shell.Settings.IsPass03 )
		{
			rating = rating switch
			{
				0 => 0,
				1 => 0,
				2 => 1,
				3 => 2,
				4 => 3,

				_ => rating
			};
		}

		return rating?.ToString();
	}

	// Returns list of filter objects for this dimension
	public static List<object> GetFilter( List<object> objects )
	{
		return objects;
	}

	// Determines whether specified rating should be included in filter
	public static bool Filter( List<object> filter, Stat stat )
	{
		if ( Include( stat ) )
		{
			KeyRating rating = new( stat );

			return (filter == null) || filter.Contains( rating );
		}

		return false;
	}

	// Determines if specified stat relevant for rating dimension
	private static bool Include( Stat stat )
	{
		// Third ball attack only
		return ((stat.Action == Stats.ThirdKey) && (stat.Selector == Stats.AttackKey) && (stat.Value != null));
	}

	// Returns comma-delimited list of selected filter values
	public static string GetValues( List<object> filter )
	{
		int count = filter.Count;
		string[] values = new string[count];

		// Build comma delimited list
		for ( int i = 0; i < count; i++ )
		{
			KeyRating rating = filter[i] as KeyRating;

			values[i] = rating?.ObjectName;
		}

		return string.Join( ", ", values );
	}

	// Loads filter tree for dimension
	public static void Load( DXCheckboxNode tree, List<object> filter )
	{
		tree.SetChildObjects( filter );
	}

	// Updates filter list for dimension
	public static List<object> Update( DataStats stats )
	{
		List<KeyRating> list = new();

		// Build list of each rating present in data
		foreach ( Stat stat in stats )
		{
			if ( Include( stat ) )
			{
				KeyRating rating = new( stat );

				if ( !list.Contains( rating ) )
				{
					list.Add( rating );
				}
			}
		}

		// Sort
		return list.OrderBy( kr => kr.ObjectName ).Cast<object>().ToList();
	}
}

//
