﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI.Control.Checkbox;
using DXLib.Utils;

namespace iStatVball3;

/*
 * Provides utility methods for working with an analyze By Modifier (contact type) key. Note that the KeyModifier object
 * is the actual key.
 */
public class KeyPosition
{
	/* Properties */
	public string UniqueId { get; set; }
	public string ObjectName { get; set; }

	/* Methods */
	public KeyPosition( Stat stat )
	{
		// 'oh', 'opp'
		UniqueId = GetKey( stat );

		// Read display name from LUT
		ObjectName = GetLabel( stat );
	}

	// Tests for equality between two modifier objects
	public override bool Equals( object obj )
	{
		return (obj is KeyPosition position) && position.UniqueId.Equals( UniqueId );
	}

	// Generates unique hash code for lists
	public override int GetHashCode()
	{
		return UniqueId.GetHashCode();
	}

	// Returns unique key for dimension analysis
	public static string GetKey( Stat stat )
	{
		return stat.Position;
	}

	// Creates display label for specified position ('oh','opp')
	public static string GetLabel( Stat stat )
	{
		string position = stat.Position;

		// Read from LUT
		return (position == null) ? null : DXString.GetLookupValue( "lineup.position", position );
	}

	// Returns list of filter objects for this dimension
	public static List<object> GetFilter( List<object> objects )
	{
		return objects;
	}

	// Determines if position from specified stat passes given filter
	public static bool Filter( List<object> filter, Stat stat )
	{
		return (filter == null) || ((stat.Position != null) && filter.Contains( new KeyPosition( stat ) ));
	}

	// Returns comma-delimited list of selected filter values
	public static string GetValues( List<object> filter )
	{
		int count = filter.Count;
		string[] values = new string[count];

		// Build comma delimited list
		for ( int i = 0; i < count; i++ )
		{
			object obj = filter[i];

			values[i] = (obj as KeyPosition)?.ObjectName;
		}

		return string.Join( ", ", values );
	}

	// Loads filter tree for dimension
	public static void Load( DXCheckboxNode tree, List<object> filter )
	{
		tree.SetChildObjects( filter );
	}

	// Updates filter list for dimension
	public static List<object> Update( DataStats stats )
	{
		List<KeyPosition> list = [];

		// Build list of each position present in data
		foreach ( Stat stat in stats )
		{
			if ( stat.Position != null )
			{
				KeyPosition position = new( stat );

				if ( !list.Contains( position ) )
				{
					list.Add( position );
				}
			}
		}

		// Alpha sort
		return list.OrderBy( kp => kp.ObjectName ).Cast<object>().ToList();
	}
}

//
