﻿/* 
 * Copyright (c)2009-2025 DemiVision, LLC. All Rights Reserved. The information 
 * herein is the CONFIDENTIAL and PROPRIETARY information of DemiVision, LLC. 
 */

using DXLib.UI.Control.Checkbox;
using DXLib.Utils;

namespace iStatVball3;

/*
 * Provides utility methods for working with analyze dimension keys (player, rotation, etc). All dimension specific
 * functionality is handled by a corresponding key class.
 */
public static class KeyDimension
{
	/* Constants */

	// Dimension keys
	public const string NoneKey = "none";
	public const string PlayerKey = "player";
	public const string RotationKey = "rotation";
	public const string PositionKey = "position";
	public const string SetterKey = "setter";
	public const string RatingKey = "rating";
	public const string RowKey = "row";
	public const string ModifierKey = "modifier";

	// Titles
	public static readonly string AllTitle = DXString.Get( "dimension.all" );
	public static readonly string NoneTitle = DXString.Get( "dimension.none" );

	/* Methods */

	// Returns dimensions available for specified report and team
	public static List<string> GetDimensions( ReportConfig config, bool team1 )
	{
		// Not all dimensions available for opponent
		return config.Dashboard.Dimensions.Where( dimension => Include( dimension, team1 ) ).ToList();
	}

	// Determines whether opponent eligible for specified dimension
	public static bool Include( string dimension, bool team1 )
    {
        return dimension switch
        {
            RotationKey => true,
			RatingKey => true,
            ModifierKey => true,

            _ => team1,
        };
    }

	// Returns dimension title ('Player' or 'Players')
	public static string GetTitle( string key, bool plural = false )
    {
		string baseStr = DXString.Get( $"dimension.{key}" );

		// Optionally plural
		return plural ? $"{baseStr}s" : baseStr;
	}

	// Displays 'All Players' when no filters selected
	public static string GetAllTitle( string key )
    {
		string title = GetTitle( key, true );

		return $"{AllTitle} {title}";
	}

	// Returns dimension specific object from specified stat
	public static object GetObject( string dimension, Stat stat )
	{
		return dimension switch
		{
			PlayerKey => stat.Player,
			RotationKey => stat.Rotation,
			PositionKey => stat.Position,
			SetterKey => stat.Player2,
			RatingKey => stat.Value,
			RowKey => stat.CourtRow,
			ModifierKey => stat.Modifier,

			_ => null,
		};
	}

	// Creates unique key for specified dimension entry
	public static string CreateKey( string dimension, Stat stat, int number, bool sideout )
	{
        return dimension switch
        {
            PlayerKey => KeyPlayer.GetKey( stat, number ),
            RotationKey => KeyRotation.GetKey( stat, sideout ),
            PositionKey => KeyPosition.GetKey( stat ),
			SetterKey => KeySetter.GetKey( stat ),
			RatingKey => KeyRating.GetKey( stat ),
			RowKey => KeyRow.GetKey( stat ),
			ModifierKey => KeyModifier.GetKey( stat ),

            _ => null,
        };
    }
	
	// Returns display label for specific dimension entry
	public static string CreateLabel( string dimension, Stat stat, int number = 1, bool sideout = false, int maxLen = 0 )
	{
        return dimension switch
        {
            PlayerKey => KeyPlayer.GetLabel( stat, number, maxLen ),
            RotationKey => KeyRotation.GetLabel( stat, sideout ),
            PositionKey => KeyPosition.GetLabel( stat ),
			SetterKey => KeySetter.GetLabel( stat ),
			RatingKey => KeyRating.GetLabel( stat ),
			RowKey => KeyRow.GetLabel( stat ),
			ModifierKey => KeyModifier.GetLabel( stat ),

            _ => null,
        };
    }

	// Returns list of dimension objects to be filtered against
	public static List<object> GetFilter( string dimension, List<object> objects )
	{
        return dimension switch
        {
            PlayerKey => KeyPlayer.GetFilter( objects ),
            RotationKey => KeyRotation.GetFilter( objects ),
            PositionKey => KeyPosition.GetFilter( objects ),
			SetterKey => KeySetter.GetFilter( objects ),
			RatingKey => KeyRating.GetFilter( objects ),
			RowKey => KeyRow.GetFilter( objects ),
			ModifierKey => KeyModifier.GetFilter( objects ),

            _ => null,
        };
    }

    // Determines whether specified stat should be included in given filter
    public static bool Filter( string dimension, DataConfig config, List<object> filter, Stat stat, int playerNum )
    {
        return dimension switch
        {
            PlayerKey => KeyPlayer.Filter( config, filter, stat, playerNum ),
            RotationKey => KeyRotation.Filter( filter, stat ),
            PositionKey => KeyPosition.Filter( filter, stat ),
			SetterKey => KeySetter.Filter( filter, stat ),
			RatingKey => KeyRating.Filter( filter, stat ),
			RowKey => KeyRow.Filter( filter, stat ),
			ModifierKey => KeyModifier.Filter( config, filter, stat ),

            _ => true,
        };
    }

	// Returns comma-delimited list of currently selected dimension values
	public static string GetValues( string dimension, List<object> filter )
    {
        return dimension switch
        {
            PlayerKey => KeyPlayer.GetValues( filter ),
			RotationKey => KeyRotation.GetValues( filter ),
			PositionKey => KeyPosition.GetValues( filter ),
			SetterKey => KeySetter.GetValues( filter ),
			RatingKey => KeyRating.GetValues( filter ),
			RowKey => KeyRow.GetValues( filter ),
			ModifierKey => KeyModifier.GetValues( filter ),

			_ => null
        };
    }

	// Loads filter tree for specified dimension
	public static void Load( string dimension, List<object> filter, DXCheckboxNode tree, bool allowEmpty )
	{
		// Root
		tree.SetRoot( "team", DXString.GetUpper( "grid.total" ) );

		// Load dimension specified items
		switch ( dimension )
        {
			case PlayerKey: KeyPlayer.Load( tree, filter, allowEmpty ); break;
			case RotationKey: KeyRotation.Load( tree, filter ); break;
			case PositionKey: KeyPosition.Load( tree, filter ); break;
			case SetterKey: KeySetter.Load( tree, filter ); break;
			case RatingKey: KeyRating.Load( tree, filter ); break;
			case RowKey: KeyRow.Load( tree, filter ); break;
			case ModifierKey: KeyModifier.Load( tree, filter ); break;
		}
	}

	// Updates filter list for specified dimension
	public static List<object> Update( string dimension, DataConfig config, DataStats stats )
    {
		return dimension switch
		{
			PlayerKey => KeyPlayer.Update( stats ),
			RotationKey => KeyRotation.Update( stats ),
			PositionKey => KeyPosition.Update( stats ),
			SetterKey => KeySetter.Update( stats ),
			RatingKey => KeyRating.Update( stats ),
			RowKey => KeyRow.Update( stats ),
			ModifierKey => KeyModifier.Update( config, stats ),

			_ => null
		};
    }
}

//
